% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/network-methods.R
\name{make_network}
\alias{make_network}
\title{Make microbiome network (igraph)}
\usage{
make_network(physeq, type="samples", distance="jaccard", max.dist = 0.4, 
    keep.isolates=FALSE, ...)
}
\arguments{
\item{physeq}{(Required). Default \code{NULL}. 
A \code{\link{phyloseq-class}} object,
or \code{\link{otu_table-class}} object,
on which \code{g} is based. \code{phyloseq-class} recommended.}

\item{type}{(Optional). Default \code{"samples"}.
 Whether the network should be samples or taxa/OTUs.
 Supported arguments are \code{"samples"}, \code{"taxa"},
 where \code{"taxa"} indicates using the OTUs/taxaindices,
 whether they actually represent species or some other taxonomic rank.

 NOTE: not all distance methods are supported if \code{"taxa"}
 selected for type. For example, the UniFrac distance and DPCoA
 cannot be calculated for taxa-wise distances, because they use
 a taxa-wise tree as part of their calculation between samples, and
 there is no transpose-equivalent for this tree.}

\item{distance}{(Optional). Default \code{"jaccard"}.
 Any supported argument to the \code{method} parameter of the 
 \code{\link{distance}} function is supported here.
 Some distance methods, like \code{"unifrac"}, may take 
 a non-trivial amount of time to calculate, in which case
 you probably want to calculate the distance matrix separately,
 save, and then provide it as the argument to \code{distance} instead.
 See below for alternatives).

 Alternatively, if you have already calculated the sample-wise distance
 object, the resulting \code{dist}-class object
 can be provided as \code{distance} instead (see examples).

 A third alternative is to provide a function that takes 
 a sample-by-taxa matrix (typical vegan orientation)
 and returns a sample-wise distance
 matrix.}

\item{max.dist}{(Optional). Default \code{0.4}. 
The maximum ecological distance (as defined by \code{distance})
allowed between two samples to still consider them ``connected'' 
by an edge in the graphical model.}

\item{keep.isolates}{(Optional). Default \code{FALSE}. Logical.
Whether to keep isolates (un-connected samples, not microbial isolates)
in the graphical model that is returned. Default results in isolates
being removed from the object.}

\item{...}{(Optional). Additional parameters passed on to \code{\link{distance}}.}
}
\value{
A \code{igraph}-class object.
}
\description{
A specialized function for creating a network representation of microbiomes,
sample-wise or taxa-wise,
based on a user-defined ecological distance and (potentially arbitrary) threshold.
The graph is ultimately represented using the 
\code{igraph}-package.
}
\examples{
# # Example plots with Enterotype Dataset
data(enterotype)
ig <- make_network(enterotype, max.dist=0.3)
plot_network(ig, enterotype, color="SeqTech", shape="Enterotype", line_weight=0.3, label=NULL)
# 
ig1 <- make_network(enterotype, max.dist=0.2)
plot_network(ig1, enterotype, color="SeqTech", shape="Enterotype", line_weight=0.3, label=NULL)
# 
# # Three methods of choosing/providing distance/distance-method
# Provide method name available to distance() function
ig <- make_network(enterotype, max.dist=0.3, distance="jaccard")
# Provide distance object, already computed
jaccdist <- distance(enterotype, "jaccard")
ih <- make_network(enterotype, max.dist=0.3, distance=jaccdist)
# Provide "custom" function.
ii <- make_network(enterotype, max.dist=0.3, distance=function(x){vegan::vegdist(x, "jaccard")})
# The have equal results:		
all.equal(ig, ih)
all.equal(ig, ii)
# 
# Try out making a trivial "network" of the 3-sample esophagus data,
# with weighted-UniFrac as distance
data(esophagus)
ij <- make_network(esophagus, "samples", "unifrac", weighted=TRUE)
}
\seealso{
\code{\link{plot_network}}
}
