/* NVTV 3dfx TV-I2C access -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: tv_tdfx.c,v 1.22 2004/01/27 17:05:26 dthierbach Exp $
 *
 * Contents:
 *
 * Routines to control the TV and the I2C bus on a 3dfx (Voodoo) card.
 *
 */

#include "local.h" /* before everything else */
#include "xfree.h" 
#include "xf86_ansic.h"

#include "mmio.h"
#include "bitmask.h"
#include "tv_i2c.h"
#include "tv_tdfx.h"
#include "xf86i2c.h"

/* -------- Documentation -------- */

/* Additional:

vgainit0
vidcfg SST_VIDEO_2X_MODE_EN
vidpll
dacmode SST_DAC_MODE2X
screensize
stride
cursloc
startaddr
clip0/1min/max
src/dstbaseaddr


FIXME:

bt mux (S-VHS: connect a0) (currently uses CONNECT_CONVERT)
auto apply

 */

/* Registers:
 *
 * Reg 10  miscInit0 Register - includes TV blank/sync delay adjust
 *
 *    ...
 *   10:8  TV blank signal delay, in flops clocked by the 2x clock. The 
 *	   objective is to synchronize the blank signal with the data output 
 *	   by matching the CLUT delay. Default is 0x0. 
 *	   000 = 2 flops; 001 = 3 flops; ...... 111 = 9 flops 
 *  13:11  TV vsync and hsync delay, like above. Default is 0x0. 
 *  16:14  Monitor vsync and hsync delay, like above. Default is 0x0. 
 *    ...
 * 
 * Reg 14  miscInit1 Register
 *
 *    ...
 *      7  Power Down CLUT. Default is 0x0. 
 *      8  Power Down DAC. Default is 0x0. 
 *      9  Power Down Video PLL. Default is 0x0. 
 *     10  Power Down Graphics PLL. Default is 0x0. 
 *     11  Power Down Memory PLL. Default is 0x0. 
 *    ...
 *
 * Reg 24  tmuGbeInit Register - includes TV Out clock delay adjust 
 *
 *    ...
 *     15  tv_out_clk_inv - invert delayed clock. Default = 0. 
 *  19:16  tv_out_clk_del_adj - choose between 16 values of delay. 
 *         Default = 0. 
 *    ...
 *
 * Reg 28  vgaInit0 Register  (0x28) 
 *
 *    ...
 *      1  Use external video timing. This bit is used to retrieve 
 *         SYNC information through the normal VGA mechanism when the 
 *         VGA CRTC is not providing timing control. 
 *    ...

 * Reg 2c  vgaInit1 Register (0x2C)  
 *
 *    ...
 *     21  Lock Horizontal Timing - 3B4/3D4 index 0,1,2,3,4,5,1a
 *     22  Lock Vertical Timing -3B4/3D4 index 6,7 (bit 7,5,3,2 and 0), 
 *         9 10, 11 (bits 3:0), 15,16,1b. 
 *     23  Lock H2 - 0x3B4/0x3D4 index 17, bit 2 
 *    ...
 *
 * Reg 3c  vidTvOutBlankVCount  TV Out Vertical Active Start/End
 *  10: 0  tv_hsync leading edges after leading edge of tv_vsync before
 * 	   vertical active region starts.
 *  15:11  reserved
 *  26:16  tv_hsync leading edges after leading edge of tv_vsync before
 * 	   vertical active region ends.
 *  31:27  reserved
 *
 * Reg 5c  vidProcCfg
 *
 *    ...
 *      0  Mode: (0=Video Processor off, VGA mode on,
 *                1=Video Processor on, VGA mode off). 
 *         Clear and set to re-enable video processor.
 *    ...
 *      4  Half mode (1: enabled). Set for vertical double scan.
 *    ...
 *     26  Clock 2X mode: Two pixels per video clock. Should halve horizontal
 *         crt values if set. (0:1x, 1:2x)
 *    ...
 * 
 * Reg 70  vidInFormat            Video in format
 *   3: 1  VMI Video in data format Y/Cb/Cr
 * 	   (100=4:1:1, 101=YUYV 4:2:2, 110=UYVY 4:2:2)
 * 	4  VMI Video in deinterlacing mode (0=off, 1=weave interlacing)
 * 	5  VMI vmi_vsync_in polarity (1=active low)
 * 	6  VMI vmi_hsync_in polarity (1=active low)
 * 	7  VMI vmi_vactive_in polarity (1=active low)
 * 	8  TV posedge: (1=Brooktree, 0=Chrontel)
 * 	   (0 = rising /falling edge for G3B5G1B3 / R5G3R3G1 data,
 * 	    1 = falling/ rising edge for same data.)
 *  10: 9  VMI Video in buffering mode (00=single, 01=double, 10=triple buffer)
 *     11  VMI Video in buffer space type (0=linear, 1=tile)
 *     12  TV tv_vsync_in polarity (1=active low)
 *     13  TV tv_hsync_in polarity (1=active low)
 *     14  VMI interface enable
 *     15  TV out interface enable
 *     16  VMI/TV Genlock enable (0=off/slave mode, 1=on/master mode)
 *     17  VMI/TV Timing signal source (0=vga, 1=generate from device)
 * 	   Signals are: vert_extra, display_ena, vfrontporch_active,
 * 	     vbackporch_active, vblank, vga_blank_n, vga_vsync, vga_hsync.
 * 	   Generated from: vmi_vsync, vmi_hsync or tv_vsync, tv_hsync
 * 	   Always input: vmi_vactive_in
 * 	   Always output (?): tv(out)_blank
 *     18 VMI/TV Genlock source (0=VMI, 1=TV)
 *     19 TV Select. 0=vga_blank, 1=display_en used to drive output video (?)
 *     20 VMI Video in horizontal decimation mode (1=on)
 *     21 VMI Video in vertical decimation mode (1=on)
 *  31:22 reserved
 * 
 * Reg 74  vidTvOutBlankHCount    TV Out Horizontal Active Start/End
 *  10: 0  pixel clock cycles after leading edge of tv_hsync before
 * 	   horizontal active region starts.
 *  15:11  reserved
 *  26:16  pixel clock cycles after leading edge of tv_vsync before
 * 	   horizontal active region ends.
 *  31:27  reserved
 * 
 * Reg 7c  vidInDecimDeltas == vidTvOutBlankHCount ??
 *
 * Reg 78  vidSerialParallelPort  Port control, including DDC/I2C
 * 	0  out  VMI parallel interface (1=enable)
 * 	1  out  VMI chip selects (CS_N)
 * 	2  out  VMI Mode A: Data strobe (DS_N)  Mode B: Read (RD_N)
 * 	3  out  VMI Mode A: Read/Write (RW_N)   Mode B: Write (WR_N)
 * 	4  in   VMI Mode A: Data ack (DTACK_N)  Mode B: Data ready (RDY)
 * 	5  out  VMI Data output enable (0=enabled/out, 1=disabled/in)
 *  13: 6  i/o  VMI Data bus (both input and output)
 *  17:14  out  VMI Address
 *     18  out  DDC port enable (1=enable, 0=disabled (default))
 *     19  out  DDC clock write (0=DCK low, 1=DCK tristate)
 *     20  out  DDC data write (0=DDA low, 1=DCK tristate)
 *     21  in   DDC clock state 
 *     22  in   DDC data state 
 *     23  out  TV port enable (1=enable, 0=disabled (default))
 *     24  out  TV clock write (0=DCK low, 1=DCK tristate)
 *     25  out  TV data write (0=DDA low, 1=DCK tristate)
 *     26  in   TV clock state 
 *     27  in   TV data state 
 *     28  out  VMI reset (0=reset)
 *     29  out  GPIO pin 1 output
 *     30  in   GPIO pin 2 input
 *     31  out  TV out reset (1=normal, 0=reset (default))
 *
 * Reg 98  vidScreenSize 
 *  11: 0  Width of screen in pixels.
 *  23:12  Height of screen in lines.
 *         Whenever this is changed, re-enable video processor via vidProcCfg.
 *         This is displayed screen size, not virtual screen size.
 *
 */

/* CRTC:

HTotal       ---   ---  1A[0]  0[7]  0[6]  0[5]  0[4]  0[3]  0[2]  0[1]  0[0] 
HDisplay     ---   ---  1A[2]  1[7]  1[6]  1[5]  1[4]  1[3]  1[2]  1[1]  1[0] 
HStartBlank  ---   ---  1A[4]  2[7]  2[6]  2[5]  2[4]  2[3]  2[2]  2[1]  2[0] 
HEndBlank    ---   ---   ---   ---  1A[5]  5[7]  3[4]  3[3]  3[2]  3[1]  3[0]
HStartSync   ---   ---  1A[6]  4[7]  4[6]  4[5]  4[4]  4[3]  4[2]  4[1]  4[0] 
HEndSync     ---   ---   ---   ---   ---  1A[7]  5[4]  5[3]  5[2]  5[1]  5[0] 
VTotal      1B[0]  7[5]  7[0]  6[7]  6[6]  6[5]  6[4]  6[3]  6[2]  6[1]  6[0] 
VActiveEnd  1B[3]  7[6]  7[1] 12[7] 12[6] 12[5] 12[4] 12[3] 12[2] 12[1] 12[0] 
VBlankStart 1B[4]  9[5]  7[3] 15[7] 15[6] 15[5] 15[4] 15[3] 15[2] 15[1] 15[0] 
VBlankEnd    ---   ---   ---  16[7] 16[6] 16[5] 16[4] 16[3] 16[2] 16[1] 16[0] 
VSyncStart   ---   7[7]  7[2] 10[7] 10[6] 10[5] 10[4] 10[3] 10[2] 10[1] 10[0] 
VSyncEnd     ---   ---   ---   ---   ---   ---   ---  11[3] 11[2] 11[1] 11[0] 

*/

/* -------- Crt access -------- */ 

/* This is copied from vgaHW.c; and should be moved somewhere else */
/* TODO: Make this uniform with NV stuff */

#define CRT_OFFSET 0xd0

#define TDFX_WR32(p,i,d)  MMIO_OUT32(((p)->MMIOBase[0]), (i), (d))
#define TDFX_RD32(p,i)    MMIO_IN32(((p)->MMIOBase[0]), (i))

#define TDFX_AND32(p,i,d) MMIO_AND32(((p)->MMIOBase[0]), (i), (d))
#define TDFX_OR32(p,i,d)  MMIO_OR32(((p)->MMIOBase[0]), (i), (d))

#define TDFX_WR8(p,i,d)  MMIO_OUT8(((p)->MMIOBase[0]), (i), (d))
#define TDFX_RD8(p,i)    MMIO_IN8(((p)->MMIOBase[0]), (i))

/* FIXME: variable port addr */

CARD8 
readCrtTdfx (TDFXPtr hwp, CARD8 index)
{
  OUTB (hwp->PIOBase[0] + hwp->IOBase + VGA_CRTC_INDEX_OFFSET, index);
  return INB (hwp->PIOBase[0] + hwp->IOBase + VGA_CRTC_DATA_OFFSET);
}

void 
writeCrtTdfx (TDFXPtr hwp, CARD8 index, CARD8 value)
{
  OUTB (hwp->PIOBase[0] + hwp->IOBase + VGA_CRTC_INDEX_OFFSET, index);
  OUTB (hwp->PIOBase[0] + hwp->IOBase + VGA_CRTC_DATA_OFFSET, value);
}

/* from vgaHW.c, but with a TDFXPtr, not a vgaHWPtr */

static void
TDFXWriteSeq (TDFXPtr hwp, CARD8 index, CARD8 value)
{
    OUTB(hwp->PIOBase[0] - 0x300 + VGA_SEQ_INDEX, index);
    OUTB(hwp->PIOBase[0] - 0x300 + VGA_SEQ_DATA, value);
}

static CARD8
TDFXReadSeq (TDFXPtr hwp, CARD8 index)
{
    OUTB(hwp->PIOBase[0] - 0x300 + VGA_SEQ_INDEX, index);
    return INB(hwp->PIOBase[0] - 0x300 + VGA_SEQ_DATA);
}

/* -------- I2C Driver Routines -------- */

#define VIDINFORMAT 0x70           /* lm_sensor: REG2 */
#define VIDSERIALPARALLELPORT 0x78 /* lm_sensor: REG */

/* bit locations in the register */
#define DDC_ENAB	0x00040000
#define DDC_SCL_OUT	0x00080000
#define DDC_SDA_OUT	0x00100000
#define DDC_SCL_IN	0x00200000
#define DDC_SDA_IN	0x00400000
#define I2C_ENAB	0x00800000
#define I2C_SCL_OUT	0x01000000
#define I2C_SDA_OUT	0x02000000
#define I2C_SCL_IN	0x04000000
#define I2C_SDA_IN	0x08000000

/* delays */
#define CYCLE_DELAY	10
#define TIMEOUT		50

/* Bus private structure. Necessary, even though it has only one element,
   because unified destruction won't work otherwise. */

typedef struct {
  TDFXPtr pTdfx;
} TDFXTvI2CRec, *TDFXTvI2CPtr;

static void 
TDFXTvI2CGetBits(I2CBusPtr b, int *clock, int *data)
{
  TDFXTvI2CPtr priv = (TDFXTvI2CPtr) b->DriverPrivate.ptr;
  TDFXPtr pTdfx = priv->pTdfx;
  register CARD32 val;

  val = TDFX_RD32(pTdfx, VIDSERIALPARALLELPORT);

  *clock = (val & I2C_SCL_IN) != 0;
  *data  = (val & I2C_SDA_IN) != 0;
}

static void 
TDFXTvI2CPutBits(I2CBusPtr b, int clock, int data)
{
  TDFXTvI2CPtr priv = (TDFXTvI2CPtr) b->DriverPrivate.ptr;
  TDFXPtr pTdfx = priv->pTdfx;
  register CARD32 val;

  val = TDFX_RD32(pTdfx, VIDSERIALPARALLELPORT);
  if (clock)
    val |= I2C_SCL_OUT;
  else
    val &= ~I2C_SCL_OUT;
  
  if (data)
    val |= I2C_SDA_OUT;
  else
    val &= ~I2C_SDA_OUT;
  
  TDFX_WR32(pTdfx, VIDSERIALPARALLELPORT, val);
}

I2CBusPtr 
TDFXTvBusCreate (TDFXPtr pTdfx, char *name)
{
  I2CBusPtr I2CPtr;
  TDFXTvI2CPtr priv;

  I2CPtr = xf86CreateI2CBusRec();
  priv = xcalloc (1, sizeof (TDFXTvI2CRec));
  if (priv) {
    priv->pTdfx = pTdfx;
  }
  if (I2CPtr) {
    I2CPtr->BusName    = name;
    I2CPtr->scrnIndex  = -1;
    I2CPtr->I2CPutBits = TDFXTvI2CPutBits;
    I2CPtr->I2CGetBits = TDFXTvI2CGetBits;
    /* Not sure about the delays. lm_sensors has udelay = mdelay = CYCLE = 10,
     * and timeout = TIMEOUT = 50. udelay is used after each bit changes,
     * mdelay is used after bytes are send, and timeout is used when trying
     * to set the clk high. The values used are about 10x larger than the 
     * default, so it should work, hopefully.
     */
    /* FIXME: Make these shorter, and see if it still works */
    I2CPtr->AcknTimeout  = 50; /* usec, default: 5 */
    I2CPtr->HoldTime     = 10; /* usec, default: 5 */
    I2CPtr->BitTimeout   = 50; /* usec, default: 5 */
    I2CPtr->ByteTimeout  = 50; /* usec, default: 5 */
    I2CPtr->AcknTimeout  = 50; /* usec, default: 5 */
    I2CPtr->StartTimeout = 50; /* usec, default: 5 */
    I2CPtr->RiseFallTime = 10; /* usec, default: 2 */
    I2CPtr->DriverPrivate.ptr = priv; 
  }
  if (I2CPtr && priv && xf86I2CBusInit (I2CPtr)) {
    return I2CPtr;
  }
  if (I2CPtr && priv) {
    RAISE (MSG_ERROR, "Bus %s is not unique", name);
  } else {
    RAISE (MSG_ERROR, "Out of memory");
  }
  TVDestroyBus (I2CPtr);
  return NULL;
}

Bool 
TDFXTvBusInit (TDFXPtr pTdfx)
{
  pTdfx->TvChain = NULL;
  /* Must be named TV0, TV1, etc.; otherwise I2C_ID won't work */
  pTdfx->TvBus = TDFXTvBusCreate (pTdfx, "TV0");
  return (pTdfx->TvBus != NULL);
}

void 
TDFXTvBusConfig (TDFXPtr pTdfx)
{
  register CARD32 val;

  RAISE (MSG_DEBUG, "tdfx bus config");
  val = TDFX_RD32(pTdfx, VIDINFORMAT);
  val &= ~SetBit(12)  /* TV vsync active high (as for nv) */
      &  ~SetBit(13)  /* TV hsync active high (as for nv) */
      &  ~SetBit(15)  /* TV interface disable */
      &  ~SetBit(16)  /* Genlock disable */
      &  ~SetBit(17)  /* VGA signals */
      &  ~SetBit(19); /* Select vga_blank instead of display_ena */
  val |= SetBit(8);   /* Posedge = Brooktree mode */
    //      |  SetBit(19);  /* Select display_ena instead of vga_blank */
  TDFX_WR32(pTdfx, VIDINFORMAT, val);

  val = TDFX_RD32(pTdfx, VIDSERIALPARALLELPORT);
  val |= SetBit(23)  /* I2C port enable */
      |  SetBit(24)  /* SCK tri state */
      |  SetBit(25)  /* SDA tri state */
      |  SetBit(31); /* TV out reset disable */
  TDFX_WR32(pTdfx, VIDSERIALPARALLELPORT, val);
}

/* -------- CRT -------- */

void 
TDFXSetCrtRegs (TDFXPtr pTdfx, TVTdfxRegs *r)
{
  int horizDisplay    = (r->HDisplay/8)    - 1;
  int horizBlankStart = (r->HBlankStart/8) - 1;
  int horizSyncStart  = (r->HSyncStart/8)     ;
  int horizSyncEnd    = (r->HSyncEnd/8)       ;
  int horizBlankEnd   = (r->HBlankEnd/8)   - 1;
  int horizTotal      = (r->HTotal/8)      - 5;
  int vertDisplay     =  r->VDisplay       - 1;
  int vertBlankStart  =  r->VBlankStart    - 1;
  int vertSyncStart   =  r->VSyncStart        ;
  int vertSyncEnd     =  r->VSyncEnd          ;
  int vertBlankEnd    =  r->VBlankEnd      - 1;
  int vertTotal       =  r->VTotal         - 2;

  /* horizSkew = (r->HSkew/8) + 1; */

  RAISE (MSG_DEBUG, "TDFXSetCrtRegs %i %i", r->HDisplay, r->VDisplay);
#ifdef FAKE_CRTC
  FPRINTF ("\ncrt[");
#endif

  /* Unlock */
  writeCrtTdfx (pTdfx, 0x11, readCrtTdfx (pTdfx, 0x11) | 0x80);

  writeCrtTdfx (pTdfx, 0x00, Set8Bits(horizTotal));
  writeCrtTdfx (pTdfx, 0x01, Set8Bits(horizDisplay));
  writeCrtTdfx (pTdfx, 0x02, Set8Bits(horizBlankStart));
  writeCrtTdfx (pTdfx, 0x03, 
		SetBitField(horizBlankEnd,4:0,4:0) 
	      | 0 /* SetBitField(horizSkew,1:0,6:5) */
	      | SetBit(7)); 
  /* index 03; 7 vert_retrace_acces = 1 */
  writeCrtTdfx (pTdfx, 0x04, Set8Bits(horizSyncStart));
  writeCrtTdfx (pTdfx, 0x05, 
		SetBitField(horizBlankEnd,5:5,7:7)
	      | SetBitField(horizSyncEnd,4:0,4:0));
  writeCrtTdfx (pTdfx, 0x06, Set8Bits(vertTotal));
  writeCrtTdfx (pTdfx, 0x07, 
		SetBitField(vertTotal,8:8,0:0)
	      | SetBitField(vertDisplay,8:8,1:1)
	      | SetBitField(vertSyncStart,8:8,2:2)
	      | SetBitField(vertDisplay,8:8,3:3)
	      | SetBit(4) /* line_compare_8 */
	      | SetBitField(vertTotal,9:9,5:5)
	      | SetBitField(vertDisplay,9:9,6:6)
	      | SetBitField(vertSyncStart,9:9,7:7));
  writeCrtTdfx (pTdfx, 0x08, 0x00); 
  writeCrtTdfx (pTdfx, 0x09, 
		SetBitField(vertBlankStart,9:9,5:5)
	      | SetBit(6) /* line_compare_9 */
	      | 0 /* SetBitField(VScan-1,4:0,4:0) */
	      | 0 /* ((r->flags & V_DBLSCAN) ? SetBit(7) : 0x00) */ );
  /* index 09: 4:0 multi_scan (num of lines-1) */
  writeCrtTdfx (pTdfx, 0x10, Set8Bits(vertSyncStart));
  writeCrtTdfx (pTdfx, 0x11, 
		SetBitField(vertSyncEnd,3:0,3:0)
	      | SetBit(5) /* disable vert retrace intr */
	      | SetBit(8) /* keep it unlocked */);
  writeCrtTdfx (pTdfx, 0x12, Set8Bits(vertDisplay));
  writeCrtTdfx (pTdfx, 0x14, 0x00); /* underline, double word */
  writeCrtTdfx (pTdfx, 0x15, Set8Bits(vertBlankStart));
  writeCrtTdfx (pTdfx, 0x16, Set8Bits(vertBlankEnd));
  writeCrtTdfx (pTdfx, 0x18, 0xff); /* line_compare_7_0 */
  writeCrtTdfx (pTdfx, 0x1a, 
		SetBitField(horizTotal,8:8,0:0)
	      | SetBitField(horizDisplay,8:8,2:2)
	      | SetBitField(horizBlankStart,8:8,4:4)
	      | SetBitField(horizBlankEnd,6:6,5:5)
	      | SetBitField(horizSyncStart,8:8,6:6)
	      | SetBitField(horizSyncEnd,5:5,7:7));
  writeCrtTdfx (pTdfx, 0x1b, 
		SetBitField(vertTotal,10:10,0:0)
	      | SetBitField(vertDisplay,10:10,2:2)
	      | SetBitField(vertBlankStart,10:10,4:4)
	      | SetBitField(vertSyncStart,10:10,6:6));
#ifdef FAKE_CRTC
  FPRINTF ("]\n");
#endif
}

void 
TDFXGetCrtRegs (TDFXPtr pTdfx, TVTdfxRegs *r)
{
  int horizDisplay;
  int horizBlankStart;
  int horizSyncStart;
  int horizSyncEnd;
  int horizBlankEnd;
  int horizTotal;
  int vertDisplay;
  int vertBlankStart;
  int vertSyncStart;
  int vertSyncEnd;
  int vertBlankEnd;
  int vertTotal;

  RAISE (MSG_DEBUG, "TDFXGetCrtRegs [");
  horizTotal      = Set8Bits(readCrtTdfx(pTdfx, 0x00)) 
                  | SetBitField(readCrtTdfx(pTdfx, 0x1a),0:0,8:8);
  horizDisplay    = Set8Bits(readCrtTdfx(pTdfx, 0x01)) 
                  | SetBitField(readCrtTdfx(pTdfx, 0x1a),2:2,8:8);
  horizBlankStart = Set8Bits(readCrtTdfx(pTdfx, 0x02))
                  | SetBitField(readCrtTdfx(pTdfx, 0x1a),4:4,8:8);
  horizBlankEnd   = SetBitField(readCrtTdfx(pTdfx, 0x03),4:0,4:0)
                  | SetBitField(readCrtTdfx(pTdfx, 0x05),7:7,5:5)
                  | SetBitField(readCrtTdfx(pTdfx, 0x1a),5:5,6:6);
  horizSyncStart  = Set8Bits(readCrtTdfx(pTdfx, 0x04)) 
                  | SetBitField(readCrtTdfx(pTdfx, 0x1a),6:6,8:8);
  horizSyncEnd    = SetBitField(readCrtTdfx(pTdfx, 0x05),4:0,4:0)
                  | SetBitField(readCrtTdfx(pTdfx, 0x1a),7:7,5:5);

  vertTotal       = Set8Bits(readCrtTdfx(pTdfx, 0x06)) 
                  | SetBitField(readCrtTdfx(pTdfx, 0x07),0:0,8:8)
                  | SetBitField(readCrtTdfx(pTdfx, 0x07),5:5,9:9)
                  | SetBitField(readCrtTdfx(pTdfx, 0x1b),0:0,10:10);
  vertDisplay     = Set8Bits(readCrtTdfx(pTdfx, 0x12)) 
	          | SetBitField(readCrtTdfx(pTdfx, 0x07),1:1,8:8)
	          | SetBitField(readCrtTdfx(pTdfx, 0x07),6:6,9:9)
	          | SetBitField(readCrtTdfx(pTdfx, 0x1b),3:3,10:10);
  vertBlankStart  = Set8Bits(readCrtTdfx(pTdfx, 0x15))
	          | SetBitField(readCrtTdfx(pTdfx, 0x07),3:3,8:8)
	          | SetBitField(readCrtTdfx(pTdfx, 0x09),5:5,9:9)
	          | SetBitField(readCrtTdfx(pTdfx, 0x1b),4:4,10:10);
  vertBlankEnd    = Set8Bits(readCrtTdfx(pTdfx, 0x16));
  vertSyncStart   = Set8Bits(readCrtTdfx(pTdfx, 0x10))
	          | SetBitField(readCrtTdfx(pTdfx, 0x07),2:2,8:8)
	          | SetBitField(readCrtTdfx(pTdfx, 0x07),7:7,9:9);
  vertSyncEnd     = SetBitField(readCrtTdfx(pTdfx, 0x11),3:0,3:0);

  /* It is not clear if blank end is relative to blank start or
     total. I have chosen total here, since blank end is equal to
     total normally, anyway. */

  horizBlankEnd  |= SetBitField (horizTotal,8:7,8:7);
  while (horizBlankEnd < horizTotal) horizBlankEnd += SetBit (7);

  horizSyncEnd  |= SetBitField (horizSyncStart,8:6,8:6);
  while (horizSyncEnd < horizSyncStart) horizSyncEnd += SetBit (6);

  vertBlankEnd  |= SetBitField (vertTotal,10:8,10:8);
  while (vertBlankEnd < vertTotal) vertBlankEnd += SetBit (8);

  while (vertSyncStart < vertDisplay) vertSyncStart += SetBit (10);

  vertSyncEnd  |= SetBitField (vertSyncStart,10:4,10:4);
  while (vertSyncEnd < vertSyncStart) vertSyncEnd += SetBit (4);

  r->HDisplay    = (horizDisplay    + 1) * 8;   
  r->HBlankStart = (horizBlankStart + 1) * 8;
  r->HBlankEnd   = (horizBlankEnd   + 1) * 8;  
  r->HSyncStart	 = (horizSyncStart     ) * 8; 
  r->HSyncEnd  	 = (horizSyncEnd       ) * 8;   
  r->HTotal    	 = (horizTotal      + 5) * 8;     
     
  r->VDisplay    = vertDisplay    + 1;
  r->VBlankStart = vertBlankStart + 1;
  r->VBlankEnd   = vertBlankEnd   + 1;
  r->VSyncStart  = vertSyncStart     ;
  r->VSyncEnd    = vertSyncEnd       ;
  r->VTotal      = vertTotal      + 2;

  r->clock = 0;  

  RAISE (MSG_DEBUG, "%i %i %i %i, %i %i %i %i]", 
	   r->HDisplay, r->HSyncStart, r->HSyncEnd, r->HTotal,
	   r->VDisplay, r->VSyncStart, r->VSyncEnd, r->VTotal);
}

void 
TDFXSetTvRegs (TDFXPtr pTdfx, TVTdfxRegs *r)
{
  register CARD32 val;

  RAISE (MSG_DEBUG, "tdfx set extra v=%i/%i h=%i/%i",
	   r->tvVBlankStart, r->tvVBlankEnd, 
	   r->tvHBlankStart, r->tvHBlankEnd);
  TDFX_WR32(pTdfx, 0x3c, SetBitField (r->tvVBlankStart,10:0,10:0)
                       | SetBitField (r->tvVBlankEnd,10:0,26:16));
  TDFX_WR32(pTdfx, 0x74, SetBitField (r->tvHBlankStart,10:0,10:0)
                       | SetBitField (r->tvHBlankEnd,10:0,26:16));

  val = TDFX_RD32 (pTdfx, 0x10); /* miscInit0 */
  val &= ~SetBitField (-1,5:0,13:8);
  val |= SetBitField (r->tvBlankDelay,2:0,10:8)
       | SetBitField (r->tvSyncDelay,2:0,13:11);
  TDFX_WR32 (pTdfx, 0x10, val);

  val = TDFX_RD32(pTdfx, 0x24); /* tmuGbeInit */
  val &= ~SetBitField (-1,3:0,19:16);;
  val |= SetBitField (r->tvLatency,3:0,19:16);
  TDFX_WR32(pTdfx, 0x24, val);
}

void 
TDFXGetTvRegs (TDFXPtr pTdfx, TVTdfxRegs *r)
{
  RAISE (MSG_DEBUG, "tdfx get extra");
  r->tvVBlankStart = SetBitField (TDFX_RD32 (pTdfx, 0x3c),10:0,10:0);
  r->tvVBlankEnd   = SetBitField (TDFX_RD32 (pTdfx, 0x3c),26:16,10:0);
  r->tvHBlankStart = SetBitField (TDFX_RD32 (pTdfx, 0x74),10:0,10:0);
  r->tvHBlankEnd   = SetBitField (TDFX_RD32 (pTdfx, 0x74),26:16,10:0);
  r->tvBlankDelay  = SetBitField (TDFX_RD32 (pTdfx, 0x10),10:8,2:0);
  r->tvSyncDelay   = SetBitField (TDFX_RD32 (pTdfx, 0x10),13:11,2:0);
  r->tvLatency     = SetBitField (TDFX_RD32 (pTdfx, 0x24),19:16,3:0);
}

void 
TDFXSetAllRegs (TDFXPtr pTdfx, TVRegs *r)
{
  /* Maybe CRT regs are not needed for TV ? */
  if ((r->devFlags & DEV_MONITOR) || 
      ((r->portHost & PORT_SYNC_DIR) == PORT_SYNC_IN))
  {
    TDFXSetCrtRegs (pTdfx, &r->crtc.tdfx);
  }
  if (r->devFlags & DEV_TELEVISION) {
    TDFXSetTvRegs (pTdfx, &r->crtc.tdfx);
  }
#if 0
  {
    register CARD8 val;

    val = TDFXReadSeq (pTdfx, 0x1);
    if (r->crtc.tdfx.flags & TDFX_FLAG_DOUBLE_PIX) {
      val |= SetBit (3);
    } else {
      val |= ~SetBit (3);
    }
    TDFXWriteSeq (pTdfx, 0x1, val);
  }
#endif
  TDFX_WR32(pTdfx, 0x98, SetBitField (r->crtc.tdfx.HScreenSize,11:0,11:0)
                       | SetBitField (r->crtc.tdfx.VScreenSize,11:0,23:12));
  if (r->crtc.tdfx.flags & TDFX_FLAG_HALF_MODE) {
    TDFX_OR32 (pTdfx, 0x5c, SetBit(4));
  } else {
    TDFX_AND32 (pTdfx, 0x5c, ~SetBit(4));
  }
  if (r->crtc.tdfx.flags & TDFX_FLAG_CLOCK2X) {
    TDFX_OR32 (pTdfx, 0x5c, SetBit(26));
    TDFX_OR32 (pTdfx, 0x4c, SetBit(0));
  } else {
    TDFX_AND32 (pTdfx, 0x5c, ~SetBit(26));
    TDFX_AND32 (pTdfx, 0x4c, ~SetBit(0));
  }
}

void TDFXGetAllRegs (TDFXPtr pTdfx, TVRegs *r)
{
  TDFXGetCrtRegs (pTdfx, &r->crtc.tdfx);
  TDFXGetTvRegs (pTdfx, &r->crtc.tdfx);
  r->crtc.tdfx.HScreenSize = SetBitField (TDFX_RD32 (pTdfx, 0x98),11:0,11:0);
  r->crtc.tdfx.VScreenSize = SetBitField (TDFX_RD32 (pTdfx, 0x98),23:12,11:0);
  r->crtc.tdfx.flags = 0;
  if (GetBit(TDFX_RD32 (pTdfx, 0x5c), 26)) 
    r->crtc.tdfx.flags |= TDFX_FLAG_CLOCK2X;
}

/* -------- Device enable/disable -------- */

int 
TDFXGetDevFlags (TDFXPtr pTdfx)
{
  if (GetBit (readCrtTdfx (pTdfx, 0x70), 15)) {
    return DEV_TELEVISION;
  } else {
    return DEV_MONITOR;
  }
}

/* -------- Device routines -------- */

void 
TDFXSetTvDevice (TDFXPtr pTdfx, I2CChainPtr chain, Bool init)
{
  RAISE (MSG_DEBUG, "tdfx set dev %p %i", chain, init);
  TVSetTvEncoder (&pTdfx->tvEncoder, chain, NULL);
  tvBusOk = TRUE;
  pTdfx->tvEncoder.InitRegs (&pTdfx->tvEncoder, PORT_TDFX);
}

void 
TDFXProbeTvDevices (TDFXPtr pTdfx)
{
  if (!pTdfx) return;
  /* TDFXUpdateTvState (pTdfx); */
  tvState = TV_OFF;
  TVCheckChain (pTdfx->TvChain);
  pTdfx->TvChain = TVProbeCreateKnown (&pTdfx->TvBus, 1, NULL); 
}

/* -------- -------- */

void 
TDFXUpdateTvState (TDFXPtr pTdfx)
{
#ifndef FAKE_MMIO
  if (GetBit (TDFX_RD32(pTdfx, VIDINFORMAT), 15)) {
    tvState = TV_ON;
  } else {
    if (tvState != TV_BARS) tvState = TV_OFF;
  }
#endif
}

void 
TDFXSetTestImage (TDFXPtr pTdfx, TVEncoderRegs *r)
{
  /* FIXME chip registers ... */
  RAISE (MSG_DEBUG, "tdfx tv testimage");
  TDFXUpdateTvState (pTdfx);
  if (tvState != TV_ON) {
    /* Only do bars if not in tv-mode */
    pTdfx->tvEncoder.SetState (&pTdfx->tvEncoder, r, TV_BARS);
  }
}

void 
TDFXSetTvMode (TDFXPtr pTdfx, TVRegs *r)
{
  register CARD32 val;
  TVState oldState, newState; /* hack for now, FIXME */

  TDFXUpdateTvState (pTdfx);
  oldState = tvState;
  newState = (r->devFlags & DEV_TELEVISION) ? TV_ON : TV_OFF;
  RAISE (MSG_DEBUG, "tdfx set tv mode %i -> %i", oldState, newState);
  if (oldState == TV_ON && newState != TV_ON) 
  {
    RAISE (MSG_DEBUG, "tdfx set tv off");
    val = TDFX_RD32(pTdfx, VIDINFORMAT);
    val &= ~SetBit(15)   /* TV interface enable off */
	&  ~SetBit(16)   /* Genlock enable off */
	&  ~SetBit(17)   /* Timing from device */
	&  ~SetBit(18);  /* Genlock source TV */
    TDFX_WR32(pTdfx, VIDINFORMAT, val);
  }
  if (oldState != newState) {
    pTdfx->tvEncoder.SetState (&pTdfx->tvEncoder, &r->enc, newState);
  }
  TDFXSetAllRegs (pTdfx, r);
  if (newState != TV_OFF) 
  {
    pTdfx->tvEncoder.SetPort (&pTdfx->tvEncoder, r->portEnc);
    pTdfx->tvEncoder.SetRegs (&pTdfx->tvEncoder, &r->enc, newState);
  }
  if (oldState != TV_ON && newState == TV_ON) {
    RAISE (MSG_DEBUG, "tdfx set tv on %08lX", val);
    val = TDFX_RD32(pTdfx, VIDINFORMAT);
    val |= SetBit(15)   /* TV interface enable on */
	|  SetBit(16)   /* Genlock enable on */
	|  SetBit(18);  /* Genlock source TV */
    /* FIXME: If changing sync polarity, should do it the right 
       way round, and allow a delay to let the lines become tristate */
    if ((r->portHost & PORT_SYNC_DIR) == PORT_SYNC_IN) {
      val &= ~SetBit(17); /* Timing from vga */
    } else {
      val |= SetBit(17);  /* Timing from device */
    }
    TDFX_WR32(pTdfx, VIDINFORMAT, val);
    val = TDFX_RD32(pTdfx, 0x24); /* tmuGbeInit */
    if ((r->portHost & PORT_PCLK_POLARITY) == PORT_PCLK_LOW) {
      val |= SetBit(15);  /* invert tv_out_clk */
    } else {
      val &= ~SetBit(15); /* normal tv_out_clk */
    }
    TDFX_WR32(pTdfx, 0x24, val);
  }
  tvState = newState;
  /* Reset video processor */
  TDFX_AND32 (pTdfx, 0x5c, ~SetBit(0));
  xf86usleep(1000); /* guess */
  TDFX_OR32 (pTdfx, 0x5c, SetBit(0));
}

long 
TDFXGetTvStatus (TDFXPtr pTdfx, int index)

{
  long result; 

  TDFXUpdateTvState (pTdfx);
  tvBusOk = TRUE;
  result = pTdfx->tvEncoder.GetStatus (&pTdfx->tvEncoder, index);
  return result;
}

TVConnect 
TDFXGetTvConnect (TDFXPtr pTdfx)
{
  TVConnect result;

  TDFXUpdateTvState (pTdfx);
  tvBusOk = TRUE;
  result = pTdfx->tvEncoder.GetConnect (&pTdfx->tvEncoder);
  return result;
}
