/*
 * Copyright (c) 2008 INRIA
 *
 * SPDX-License-Identifier: GPL-2.0-only
 *
 * Author: Mathieu Lacage <mathieu.lacage@sophia.inria.fr>
 */

#ifndef SPECTRUM_WIFI_HELPER_H
#define SPECTRUM_WIFI_HELPER_H

#include "wifi-helper.h"

#include <map>
#include <set>

namespace ns3
{

class SpectrumChannel;
class SpectrumWifiPhy;

/**
 * @brief Make it easy to create and manage PHY objects for the spectrum model.
 *
 * The Pcap and ASCII traces generated by the EnableAscii and EnablePcap methods defined
 * in this class correspond to PHY-level traces and come to us via WifiPhyHelper
 *
 * This helper will install an instance of a WifiBandwidthFilter on any SpectrumChannel
 * added to this helper (via one of the SetChannel methods), unless one has
 * previously been installed on the channel object.
 */
class SpectrumWifiPhyHelper : public WifiPhyHelper
{
  public:
    /**
     * Create a PHY helper.
     *
     * @param nLinks the number of links to configure (>1 only for 11be devices)
     */
    SpectrumWifiPhyHelper(uint8_t nLinks = 1);

    /**
     * @param channel the default spectrum channel to associate to this helper
     *
     * Every PHY created by a call to Install is associated to this default spectrum channel.
     */
    void SetChannel(const Ptr<SpectrumChannel> channel);
    /**
     * @param channelName The name of the default spectrum channel to associate to this helper
     *
     * Every PHY created by a call to Install is associated to this default spectrum channel.
     */
    void SetChannel(const std::string& channelName);

    /**
     * @param channel the spectrum channel to add to this helper
     * @param freqRange the frequency range, bounded by a minFrequency and a maxFrequency in MHz
     *
     * Every PHY created by a call to Install is added to this spectrum channel.
     * If a PHY is requested to operate or scan a channel with the specified frequency and width
     * combination, it will activate that channel and deactivate the current channel for that PHY.
     */
    void AddChannel(const Ptr<SpectrumChannel> channel,
                    const FrequencyRange& freqRange = WHOLE_WIFI_SPECTRUM);
    /**
     * @param channelName The name of the spectrum channel to add to this helper
     * @param freqRange the frequency range, bounded by a minFrequency and a maxFrequency in MHz
     *
     * Every PHY created by a call to Install is added to this spectrum channel.
     * If a PHY is requested to operate or scan a channel with the specified frequency and width
     * combination, it will activate that channel and deactivate the current channel for that PHY.
     */
    void AddChannel(const std::string& channelName,
                    const FrequencyRange& freqRange = WHOLE_WIFI_SPECTRUM);

    /**
     * Add a given spectrum PHY interface to the PHY instance corresponding of a given link.
     * If no mapping has been specified for a given link, all spectrum PHY interfaces will
     * be added to the PHY instance of that link.
     *
     * @param linkId ID of the link to setup
     * @param freqRange frequency range handled by of the spectrum PHY interface
     */
    void AddPhyToFreqRangeMapping(uint8_t linkId, const FrequencyRange& freqRange);

    /**
     * Reset mapping of the spectrum PHY interfaces added to the PHY instances.
     */
    void ResetPhyToFreqRangeMapping();

  private:
    /**
     * @param node the node on which we wish to create a wifi PHY
     * @param device the device within which this PHY will be created
     * @returns newly-created PHY objects.
     *
     * This method implements the pure virtual method defined in \ref ns3::WifiPhyHelper.
     */
    std::vector<Ptr<WifiPhy>> Create(Ptr<Node> node, Ptr<WifiNetDevice> device) const override;

    /**
     * @brief Install PHY interfaces to the PHY instance of a given link
     * based on the currently configured mapping (\see AddPhyInterface).
     *
     * @param linkId ID of the link to setup
     * @param phy spectrum PHY instance of the link
     */
    void InstallPhyInterfaces(uint8_t linkId, Ptr<SpectrumWifiPhy> phy) const;

    /**
     * @param channel The channel to inspect to possibly add a WifiBandwidthFilter
     *
     * This method will add a WifiBandwidthFilter object to the SpectrumChannel, only if
     * one has not yet been added.
     */
    void AddWifiBandwidthFilter(Ptr<SpectrumChannel> channel);

    /**
     * Function that is notified when a spectrum channel switched
     * @param phy spectrum PHY instance that has switched its channel
     */
    static void SpectrumChannelSwitched(Ptr<SpectrumWifiPhy> phy);

    std::map<FrequencyRange, Ptr<SpectrumChannel>> m_channels; ///< the spectrum channels
    std::map<uint8_t /* linkId */, std::set<FrequencyRange>>
        m_interfacesMap; ///< map of the spectrum PHY interfaces to be added to the PHY instance
                         ///< corresponding to a given link ID
};

} // namespace ns3

#endif /* SPECTRUM_WIFI_HELPER_H */
