/*
 * xtc - The eXTensible Compiler
 * Copyright (C) 2004-2007 Robert Grimm
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
 * USA.
 */
package xtc.util;

/**
 * The interface for a global parser state object.
 *
 * <p />To correctly integrate with a memoizing parser, global state
 * for parsers generated by <i>Rats!</i> is modified through
 * light-weight, nested transactions, as expressed through this
 * interface.

 * <p />A grammar utilizing global state needs to have a global {@link
 * xtc.Constants#ATT_STATEFUL stateful} attribute, whose value is the
 * name of the class implementing this interface.  The class, in turn,
 * must have a no-argument constructor, which is used to create the
 * global state object.
 *
 * <p />Each production that resets the global state (i.e., serves as
 * a top-level entry point) needs to be marked with the {@link
 * xtc.Constants#ATT_RESETTING resetting} attribute.  At the beginning
 * of the method representing such a production, the global state is
 * reset by calling {@link #reset(String)}.
 *
 * <p />Each production that might modify the global state (or that
 * depends on other productions that modify the global state) needs to
 * be marked with the {@link xtc.Constants#ATT_STATEFUL stateful}
 * attribute.  At the beginning of the method representing such a
 * production, a new transaction is started by calling {@link
 * #start()}.  This transaction is completed on a successful parse by
 * calling {@link #commit()} and on an erroneous parse by calling
 * {@link #abort()}.
 *
 * @author Robert Grimm
 * @version $Revision: 1.9 $
 */
public interface State {

  /**
   * Reset the global state object.  This method is called at the
   * beginning of each method representing a production with the
   * <code>resetting</code> attribute.
   *
   * @param file The file name.
   */
  void reset(String file);

  /**
   * Start a new state-modifying transaction.  This method is called
   * at the beginning of each method representing a production with
   * the <code>stateful</code> attribute.
   */
  void start();

  /**
   * Commit a state-modifying transaction.  This method is called on a
   * successful parse before returning from a method representing a
   * production with the <code>stateful</code> attribute.
   */
  void commit();

  /**
   * Abort a state-modifying transaction.  This method is called on an
   * erroneous parse before returning from a method representing a
   * production with the <code>stateful</code> attribute.
   */
  void abort();

}
