/* GStreamer unit tests for flvmux
 *
 * Copyright (C) 2009 Tim-Philipp Müller  <tim centricular net>
 * Copyright (C) 2016 Havard Graff <havard@pexip.com>
 * Copyright (C) 2016 David Buchmann <david@pexip.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_VALGRIND
# include <valgrind/valgrind.h>
#endif

#include <gst/check/gstcheck.h>
#include <gst/check/gstharness.h>

#include <gst/gst.h>

static GstBusSyncReply
error_cb (GstBus * bus, GstMessage * msg, gpointer user_data)
{
  if (GST_MESSAGE_TYPE (msg) == GST_MESSAGE_ERROR) {
    GError *err = NULL;
    gchar *dbg = NULL;

    gst_message_parse_error (msg, &err, &dbg);
    g_error ("ERROR: %s\n%s\n", err->message, dbg);
  }

  return GST_BUS_PASS;
}

static void
handoff_cb (GstElement * element, GstBuffer * buf, GstPad * pad,
    gint * p_counter)
{
  *p_counter += 1;
  GST_LOG ("counter = %d", *p_counter);
}

static void
mux_pcm_audio (guint num_buffers, guint repeat)
{
  GstElement *src, *sink, *flvmux, *conv, *pipeline;
  GstPad *sinkpad, *srcpad;
  gint counter;

  GST_LOG ("num_buffers = %u", num_buffers);

  pipeline = gst_pipeline_new ("pipeline");
  fail_unless (pipeline != NULL, "Failed to create pipeline!");

  /* kids, don't use a sync handler for this at home, really; we do because
   * we just want to abort and nothing else */
  gst_bus_set_sync_handler (GST_ELEMENT_BUS (pipeline), error_cb, NULL, NULL);

  src = gst_element_factory_make ("audiotestsrc", "audiotestsrc");
  fail_unless (src != NULL, "Failed to create 'audiotestsrc' element!");

  g_object_set (src, "num-buffers", num_buffers, NULL);

  conv = gst_element_factory_make ("audioconvert", "audioconvert");
  fail_unless (conv != NULL, "Failed to create 'audioconvert' element!");

  flvmux = gst_element_factory_make ("flvmux", "flvmux");
  fail_unless (flvmux != NULL, "Failed to create 'flvmux' element!");

  sink = gst_element_factory_make ("fakesink", "fakesink");
  fail_unless (sink != NULL, "Failed to create 'fakesink' element!");

  g_object_set (sink, "signal-handoffs", TRUE, NULL);
  g_signal_connect (sink, "handoff", G_CALLBACK (handoff_cb), &counter);

  gst_bin_add_many (GST_BIN (pipeline), src, conv, flvmux, sink, NULL);

  fail_unless (gst_element_link (src, conv));
  fail_unless (gst_element_link (flvmux, sink));

  /* now link the elements */
  sinkpad = gst_element_request_pad_simple (flvmux, "audio");
  fail_unless (sinkpad != NULL, "Could not get audio request pad");

  srcpad = gst_element_get_static_pad (conv, "src");
  fail_unless (srcpad != NULL, "Could not get audioconvert's source pad");

  fail_unless_equals_int (gst_pad_link (srcpad, sinkpad), GST_PAD_LINK_OK);

  gst_object_unref (srcpad);
  gst_object_unref (sinkpad);

  do {
    GstStateChangeReturn state_ret;
    GstMessage *msg;

    GST_LOG ("repeat=%d", repeat);

    counter = 0;

    state_ret = gst_element_set_state (pipeline, GST_STATE_PAUSED);
    fail_unless (state_ret != GST_STATE_CHANGE_FAILURE);

    if (state_ret == GST_STATE_CHANGE_ASYNC) {
      GST_LOG ("waiting for pipeline to reach PAUSED state");
      state_ret = gst_element_get_state (pipeline, NULL, NULL, -1);
      fail_unless_equals_int (state_ret, GST_STATE_CHANGE_SUCCESS);
    }

    GST_LOG ("PAUSED, let's do the rest of it");

    state_ret = gst_element_set_state (pipeline, GST_STATE_PLAYING);
    fail_unless (state_ret != GST_STATE_CHANGE_FAILURE);

    msg = gst_bus_poll (GST_ELEMENT_BUS (pipeline), GST_MESSAGE_EOS, -1);
    fail_unless (msg != NULL, "Expected EOS message on bus!");

    GST_LOG ("EOS");
    gst_message_unref (msg);

    /* should have some output */
    fail_unless (counter > 2);

    fail_unless_equals_int (gst_element_set_state (pipeline, GST_STATE_NULL),
        GST_STATE_CHANGE_SUCCESS);

    /* repeat = test re-usability */
    --repeat;
  } while (repeat > 0);

  gst_object_unref (pipeline);
}

GST_START_TEST (test_index_writing)
{
  /* note: there's a magic 128 value in flvmux when doing index writing */
  mux_pcm_audio (__i__ * 33 + 1, 2);
}

GST_END_TEST;

static GstBuffer *
create_buffer (guint8 * data, gsize size,
    GstClockTime timestamp, GstClockTime duration)
{
  GstBuffer *buf = gst_buffer_new_wrapped_full (GST_MEMORY_FLAG_READONLY,
      data, size, 0, size, NULL, NULL);
  GST_BUFFER_PTS (buf) = timestamp;
  GST_BUFFER_DTS (buf) = timestamp;
  GST_BUFFER_DURATION (buf) = duration;
  GST_BUFFER_OFFSET (buf) = 0;
  GST_BUFFER_OFFSET_END (buf) = 0;
  return buf;
}

guint8 speex_hdr0[] = {
  0x53, 0x70, 0x65, 0x65, 0x78, 0x20, 0x20, 0x20,
  0x31, 0x2e, 0x32, 0x72, 0x63, 0x31, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
  0x50, 0x00, 0x00, 0x00, 0x80, 0x3e, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff,
  0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

guint8 speex_hdr1[] = {
  0x1f, 0x00, 0x00, 0x00, 0x45, 0x6e, 0x63, 0x6f,
  0x64, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68,
  0x20, 0x47, 0x53, 0x74, 0x72, 0x65, 0x61, 0x6d,
  0x65, 0x72, 0x20, 0x53, 0x70, 0x65, 0x65, 0x78,
  0x65, 0x6e, 0x63, 0x00, 0x00, 0x00, 0x00, 0x01
};

guint8 speex_buf[] = {
  0x36, 0x9d, 0x1b, 0x9a, 0x20, 0x00, 0x01, 0x68,
  0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0xe8, 0x84,
  0x00, 0xb4, 0x74, 0x74, 0x74, 0x74, 0x74, 0x74,
  0x74, 0x42, 0x00, 0x5a, 0x3a, 0x3a, 0x3a, 0x3a,
  0x3a, 0x3a, 0x3a, 0x21, 0x00, 0x2d, 0x1d, 0x1d,
  0x1d, 0x1d, 0x1d, 0x1d, 0x1d, 0x1b, 0x3b, 0x60,
  0xab, 0xab, 0xab, 0xab, 0xab, 0x0a, 0xba, 0xba,
  0xba, 0xba, 0xb0, 0xab, 0xab, 0xab, 0xab, 0xab,
  0x0a, 0xba, 0xba, 0xba, 0xba, 0xb7
};

guint8 h264_buf[] = {
  0x00, 0x00, 0x00, 0x0b, 0x67, 0x42, 0xc0, 0x0c,
  0x95, 0xa7, 0x20, 0x1e, 0x11, 0x08, 0xd4, 0x00,
  0x00, 0x00, 0x04, 0x68, 0xce, 0x3c, 0x80, 0x00,
  0x00, 0x00, 0x55, 0x65, 0xb8, 0x04, 0x0e, 0x7e,
  0x1f, 0x22, 0x60, 0x34, 0x01, 0xe2, 0x00, 0x3c,
  0xe1, 0xfc, 0x91, 0x40, 0xa6, 0x9e, 0x07, 0x42,
  0x56, 0x44, 0x73, 0x75, 0x40, 0x9f, 0x0c, 0x87,
  0x83, 0xc9, 0x52, 0x60, 0x6d, 0xd8, 0x98, 0x01,
  0x16, 0xbd, 0x0f, 0xa6, 0xaf, 0x75, 0x83, 0xdd,
  0xfa, 0xe7, 0x8f, 0xe3, 0x58, 0x10, 0x0f, 0x5c,
  0x18, 0x2f, 0x41, 0x40, 0x23, 0x0b, 0x03, 0x70,
  0x00, 0xff, 0xe4, 0xa6, 0x7d, 0x7f, 0x3f, 0x76,
  0x01, 0xd0, 0x98, 0x2a, 0x0c, 0xb8, 0x02, 0x32,
  0xbc, 0x56, 0xfd, 0x34, 0x4f, 0xcf, 0xfe, 0xa0,
};

GST_START_TEST (test_speex_streamable)
{
  GstBuffer *buf;
  GstMapInfo map = GST_MAP_INFO_INIT;


  GstCaps *caps = gst_caps_new_simple ("audio/x-speex",
      "rate", G_TYPE_INT, 16000,
      "channels", G_TYPE_INT, 1,
      NULL);

  const GstClockTime base_time = 123456789;
  const GstClockTime duration_ms = 20;
  const GstClockTime duration = duration_ms * GST_MSECOND;

  GstHarness *h = gst_harness_new_with_padnames ("flvmux", "audio", "src");
  gst_harness_set_src_caps (h, caps);
  g_object_set (h->element, "streamable", 1, NULL);

  /* push speex header0 */
  gst_harness_push (h, create_buffer (speex_hdr0,
          sizeof (speex_hdr0), base_time, 0));

  /* push speex header1 */
  gst_harness_push (h, create_buffer (speex_hdr1,
          sizeof (speex_hdr1), base_time, 0));

  /* push speex data */
  gst_harness_push (h, create_buffer (speex_buf,
          sizeof (speex_buf), base_time, duration));

  /* push speex data 2 */
  gst_harness_push (h, create_buffer (speex_buf,
          sizeof (speex_buf), base_time + duration, duration));

  /* pull out stream-start event */
  gst_event_unref (gst_harness_pull_event (h));

  /* pull out caps event */
  gst_event_unref (gst_harness_pull_event (h));

  /* pull out segment event and verify we are using GST_FORMAT_TIME */
  {
    GstEvent *event = gst_harness_pull_event (h);
    const GstSegment *segment;
    gst_event_parse_segment (event, &segment);
    fail_unless_equals_int (GST_FORMAT_TIME, segment->format);
    gst_event_unref (event);
  }

  /* pull FLV header buffer */
  buf = gst_harness_pull (h);
  gst_buffer_unref (buf);

  /* pull Metadata buffer */
  buf = gst_harness_pull (h);
  gst_buffer_unref (buf);

  /* pull header0 */
  buf = gst_harness_pull (h);
  fail_unless_equals_uint64 (base_time, GST_BUFFER_PTS (buf));
  fail_unless_equals_uint64 (GST_CLOCK_TIME_NONE, GST_BUFFER_DTS (buf));
  fail_unless_equals_uint64 (0, GST_BUFFER_DURATION (buf));
  gst_buffer_map (buf, &map, GST_MAP_READ);
  /* 0x08 means it is audio */
  fail_unless_equals_int (0x08, map.data[0]);
  /* timestamp should be starting from 0 */
  fail_unless_equals_int (0x00, map.data[6]);
  /* 0xb2 means Speex, 16000Hz, Mono */
  fail_unless_equals_int (0xb2, map.data[11]);
  /* verify content is intact */
  fail_unless_equals_int (0, memcmp (&map.data[12], speex_hdr0,
          sizeof (speex_hdr0)));
  gst_buffer_unmap (buf, &map);
  gst_buffer_unref (buf);

  /* pull header1 */
  buf = gst_harness_pull (h);
  fail_unless_equals_uint64 (base_time, GST_BUFFER_PTS (buf));
  fail_unless_equals_uint64 (GST_CLOCK_TIME_NONE, GST_BUFFER_DTS (buf));
  fail_unless_equals_uint64 (0, GST_BUFFER_DURATION (buf));
  gst_buffer_map (buf, &map, GST_MAP_READ);
  /* 0x08 means it is audio */
  fail_unless_equals_int (0x08, map.data[0]);
  /* timestamp should be starting from 0 */
  fail_unless_equals_int (0x00, map.data[6]);
  /* 0xb2 means Speex, 16000Hz, Mono */
  fail_unless_equals_int (0xb2, map.data[11]);
  /* verify content is intact */
  fail_unless_equals_int (0, memcmp (&map.data[12], speex_hdr1,
          sizeof (speex_hdr1)));
  gst_buffer_unmap (buf, &map);
  gst_buffer_unref (buf);

  /* pull data */
  buf = gst_harness_pull (h);
  fail_unless_equals_uint64 (base_time, GST_BUFFER_PTS (buf));
  fail_unless_equals_uint64 (GST_CLOCK_TIME_NONE, GST_BUFFER_DTS (buf));
  fail_unless_equals_uint64 (duration, GST_BUFFER_DURATION (buf));
  fail_unless_equals_uint64 (GST_BUFFER_OFFSET_NONE, GST_BUFFER_OFFSET (buf));
  fail_unless_equals_uint64 (GST_BUFFER_OFFSET_NONE,
      GST_BUFFER_OFFSET_END (buf));
  gst_buffer_map (buf, &map, GST_MAP_READ);
  /* 0x08 means it is audio */
  fail_unless_equals_int (0x08, map.data[0]);
  /* timestamp should be starting from 0 */
  fail_unless_equals_int (0x00, map.data[6]);
  /* 0xb2 means Speex, 16000Hz, Mono */
  fail_unless_equals_int (0xb2, map.data[11]);
  /* verify content is intact */
  fail_unless_equals_int (0, memcmp (&map.data[12], speex_buf,
          sizeof (speex_buf)));
  gst_buffer_unmap (buf, &map);
  gst_buffer_unref (buf);

  /* pull data */
  buf = gst_harness_pull (h);
  fail_unless_equals_uint64 (base_time + duration, GST_BUFFER_PTS (buf));
  fail_unless_equals_uint64 (GST_CLOCK_TIME_NONE, GST_BUFFER_DTS (buf));
  fail_unless_equals_uint64 (duration, GST_BUFFER_DURATION (buf));
  fail_unless_equals_uint64 (GST_BUFFER_OFFSET_NONE, GST_BUFFER_OFFSET (buf));
  fail_unless_equals_uint64 (GST_BUFFER_OFFSET_NONE,
      GST_BUFFER_OFFSET_END (buf));
  gst_buffer_map (buf, &map, GST_MAP_READ);
  /* 0x08 means it is audio */
  fail_unless_equals_int (0x08, map.data[0]);
  /* timestamp should reflect the duration_ms */
  fail_unless_equals_int (duration_ms, map.data[6]);
  /* 0xb2 means Speex, 16000Hz, Mono */
  fail_unless_equals_int (0xb2, map.data[11]);
  /* verify content is intact */
  fail_unless_equals_int (0, memcmp (&map.data[12], speex_buf,
          sizeof (speex_buf)));
  gst_buffer_unmap (buf, &map);
  gst_buffer_unref (buf);

  gst_harness_teardown (h);
}

GST_END_TEST;

static void
check_buf_type_timestamp (GstBuffer * buf, gint packet_type, gint timestamp)
{
  GstMapInfo map = GST_MAP_INFO_INIT;
  gst_buffer_map (buf, &map, GST_MAP_READ);
  fail_unless_equals_int (packet_type, map.data[0]);
  fail_unless_equals_int (timestamp, map.data[6]);
  gst_buffer_unmap (buf, &map);
  gst_buffer_unref (buf);
}

static const gint AUDIO = 0x08;
static const gint VIDEO = 0x09;

GST_START_TEST (test_increasing_timestamp_when_pts_none)
{
  gint timestamp = 3;
  GstClockTime base_time = 42 * GST_SECOND;
  GstPad *audio_sink, *video_sink, *audio_src, *video_src;
  GstHarness *h, *audio, *video, *audio_q, *video_q;
  GstCaps *audio_caps, *video_caps;
  GstBuffer *buf;

  h = gst_harness_new_with_padnames ("flvmux", NULL, "src");
  audio = gst_harness_new_with_element (h->element, "audio", NULL);
  video = gst_harness_new_with_element (h->element, "video", NULL);
  audio_q = gst_harness_new ("queue");
  video_q = gst_harness_new ("queue");

  audio_sink = GST_PAD_PEER (audio->srcpad);
  video_sink = GST_PAD_PEER (video->srcpad);
  audio_src = GST_PAD_PEER (audio_q->sinkpad);
  video_src = GST_PAD_PEER (video_q->sinkpad);

  gst_pad_unlink (audio->srcpad, audio_sink);
  gst_pad_unlink (video->srcpad, video_sink);
  gst_pad_unlink (audio_src, audio_q->sinkpad);
  gst_pad_unlink (video_src, video_q->sinkpad);
  gst_pad_link (audio_src, audio_sink);
  gst_pad_link (video_src, video_sink);

  audio_caps = gst_caps_new_simple ("audio/x-speex",
      "rate", G_TYPE_INT, 16000, "channels", G_TYPE_INT, 1, NULL);
  gst_harness_set_src_caps (audio_q, audio_caps);
  video_caps = gst_caps_new_simple ("video/x-h264",
      "stream-format", G_TYPE_STRING, "avc", NULL);
  gst_harness_set_src_caps (video_q, video_caps);

  /* Push audio + video + audio with increasing DTS, but PTS for video is
   * GST_CLOCK_TIME_NONE
   */
  buf = gst_buffer_new ();
  GST_BUFFER_DTS (buf) = timestamp * GST_MSECOND + base_time;
  GST_BUFFER_PTS (buf) = timestamp * GST_MSECOND + base_time;
  gst_harness_push (audio_q, buf);

  buf = gst_buffer_new ();
  GST_BUFFER_DTS (buf) = (timestamp + 1) * GST_MSECOND + base_time;
  GST_BUFFER_PTS (buf) = GST_CLOCK_TIME_NONE;
  gst_harness_push (video_q, buf);

  buf = gst_buffer_new ();
  GST_BUFFER_DTS (buf) = (timestamp + 2) * GST_MSECOND + base_time;
  GST_BUFFER_PTS (buf) = (timestamp + 2) * GST_MSECOND + base_time;
  gst_harness_push (audio_q, buf);

  /* Pull two metadata packets out */
  gst_buffer_unref (gst_harness_pull (h));
  gst_buffer_unref (gst_harness_pull (h));

  /* Check that we receive the packets in monotonically increasing order and
   * that their timestamps are correct (should start at 0)
   */
  buf = gst_harness_pull (h);
  check_buf_type_timestamp (buf, AUDIO, 0);
  buf = gst_harness_pull (h);
  check_buf_type_timestamp (buf, VIDEO, 1);

  /* teardown */
  gst_harness_teardown (h);
  gst_harness_teardown (audio);
  gst_harness_teardown (video);
  gst_harness_teardown (audio_q);
  gst_harness_teardown (video_q);
}

GST_END_TEST;

typedef struct
{
  GstHarness *a_sink;
  GstHarness *v_sink;
} DemuxHarnesses;

static void
flvdemux_pad_added (GstElement * flvdemux, GstPad * srcpad, DemuxHarnesses * h)
{
  GstCaps *caps = gst_pad_get_current_caps (srcpad);
  const gchar *name = gst_structure_get_name (gst_caps_get_structure (caps, 0));

  if (h->a_sink && g_ascii_strncasecmp ("audio", name, 5) == 0)
    gst_harness_add_element_src_pad (h->a_sink, srcpad);
  else if (h->v_sink && g_ascii_strncasecmp ("video", name, 5) == 0)
    gst_harness_add_element_src_pad (h->v_sink, srcpad);
  else
    ck_abort_msg ("Unexpected demux pad: %s", GST_STR_NULL (name));

  gst_caps_unref (caps);
}

GST_START_TEST (test_video_caps_late)
{
  GstHarness *mux = gst_harness_new_with_padnames ("flvmux", NULL, "src");
  GstHarness *a_src =
      gst_harness_new_with_element (mux->element, "audio", NULL);
  GstHarness *v_src =
      gst_harness_new_with_element (mux->element, "video", NULL);
  GstHarness *demux = gst_harness_new_with_padnames ("flvdemux", "sink", NULL);
  GstHarness *a_sink =
      gst_harness_new_with_element (demux->element, NULL, NULL);
  GstHarness *v_sink =
      gst_harness_new_with_element (demux->element, NULL, NULL);
  DemuxHarnesses harnesses = { a_sink, v_sink };
  GstTestClock *tclock;

  g_object_set (mux->element, "streamable", TRUE,
      "latency", G_GUINT64_CONSTANT (1), NULL);
  gst_harness_use_testclock (mux);

  g_signal_connect (demux->element, "pad-added",
      G_CALLBACK (flvdemux_pad_added), &harnesses);
  gst_harness_add_sink_harness (mux, demux);

  gst_harness_set_src_caps_str (a_src,
      "audio/x-speex, rate=(int)16000, channels=(int)1");

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (a_src,
          gst_buffer_new_wrapped_full (GST_MEMORY_FLAG_READONLY,
              speex_hdr0, sizeof (speex_hdr0), 0, sizeof (speex_hdr0), NULL,
              NULL)));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (a_src,
          gst_buffer_new_wrapped_full (GST_MEMORY_FLAG_READONLY,
              speex_hdr1, sizeof (speex_hdr1), 0, sizeof (speex_hdr1), NULL,
              NULL)));

  /* Wait a little and make sure no clock was scheduled as this shouldn't happen
   * before the caps are set */
  g_usleep (40 * 1000);
  tclock = gst_harness_get_testclock (mux);
  fail_unless (gst_test_clock_get_next_entry_time (tclock) ==
      GST_CLOCK_TIME_NONE);

  gst_harness_set_src_caps_str (v_src,
      "video/x-h264, stream-format=(string)avc, alignment=(string)au, "
      "codec_data=(buffer)0142c00cffe1000b6742c00c95a7201e1108d401000468ce3c80");

  gst_harness_crank_single_clock_wait (mux);

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (a_src,
          gst_buffer_new_wrapped_full (GST_MEMORY_FLAG_READONLY,
              speex_buf, sizeof (speex_buf), 0, sizeof (speex_buf), NULL,
              NULL)));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (v_src,
          gst_buffer_new_wrapped_full (GST_MEMORY_FLAG_READONLY,
              h264_buf, sizeof (h264_buf), 0, sizeof (h264_buf), NULL, NULL)));

  gst_harness_crank_single_clock_wait (mux);
  gst_harness_crank_single_clock_wait (mux);
  gst_harness_crank_single_clock_wait (mux);


  /* push from flvmux to demux */
  fail_unless_equals_int (GST_FLOW_OK, gst_harness_sink_push_many (mux, 7));

  /* verify we got 2x audio and 1x video buffers out of flvdemux */
  gst_buffer_unref (gst_harness_pull (a_sink));
  gst_buffer_unref (gst_harness_pull (a_sink));
  gst_buffer_unref (gst_harness_pull (v_sink));

  fail_unless (gst_test_clock_get_next_entry_time (tclock) ==
      GST_CLOCK_TIME_NONE);

  g_clear_object (&tclock);
  gst_harness_teardown (a_src);
  gst_harness_teardown (v_src);
  gst_harness_teardown (mux);
  gst_harness_teardown (a_sink);
  gst_harness_teardown (v_sink);
}

GST_END_TEST;

guint8 raw_frame_short[] = {
  0x27, 0x00, 0x03, 0x20, 0x64, 0x1c
};

GST_START_TEST (test_video_caps_change_streamable)
{
  GstEvent *event;
  GstCaps *a_caps1, *v_caps1, *v_caps2, *ret_caps;
  GstHarness *mux = gst_harness_new_with_padnames ("flvmux", NULL, "src");
  GstHarness *a_src =
      gst_harness_new_with_element (mux->element, "audio", NULL);
  GstHarness *v_src =
      gst_harness_new_with_element (mux->element, "video", NULL);
  GstHarness *demux = gst_harness_new_with_padnames ("flvdemux", "sink", NULL);
  GstHarness *a_sink =
      gst_harness_new_with_element (demux->element, NULL, NULL);
  GstHarness *v_sink =
      gst_harness_new_with_element (demux->element, NULL, NULL);
  DemuxHarnesses harnesses = { a_sink, v_sink };

  const GstClockTime base_time = 123456789;
  const GstClockTime duration_ms = 20;
  const GstClockTime duration = duration_ms * GST_MSECOND;

  g_object_set (mux->element, "streamable", TRUE, NULL);

  g_signal_connect (demux->element, "pad-added",
      G_CALLBACK (flvdemux_pad_added), &harnesses);
  gst_harness_add_sink_harness (mux, demux);

  a_caps1 =
      gst_caps_from_string
      ("audio/mpeg, mpegversion=(int)4, framed=(boolean)true, stream-format=(string)raw, "
      "rate=(int)44100, channels=(int)1, codec_data=(buffer)1208");

  v_caps1 = gst_caps_from_string ("video/x-h264, stream-format=(string)avc, "
      "codec_data=(buffer)0142c00cffe1000b6742c00c95a7201e1108d401000468ce3c80");

  gst_harness_set_src_caps (a_src, gst_caps_ref (a_caps1));
  gst_harness_set_src_caps (v_src, gst_caps_ref (v_caps1));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (a_src,
          create_buffer (raw_frame_short, sizeof (raw_frame_short), base_time,
              duration)));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (v_src,
          create_buffer (h264_buf, sizeof (h264_buf), base_time, duration)));

  gst_harness_crank_single_clock_wait (mux);

  /* push from flvmux to demux */
  fail_unless_equals_int (GST_FLOW_OK, gst_harness_sink_push_many (mux, 6));

  /* should accept without the constraint */
  while ((event = gst_harness_try_pull_event (v_sink))) {
    if (GST_EVENT_TYPE (event) == GST_EVENT_CAPS) {
      gst_event_parse_caps (event, &ret_caps);
      GST_LOG ("v_caps1 %" GST_PTR_FORMAT ", ret caps %" GST_PTR_FORMAT,
          v_caps1, ret_caps);
      fail_unless (gst_caps_is_equal (v_caps1, ret_caps));
    }
    gst_event_unref (event);
  }

  /* caps change */
  v_caps2 = gst_caps_from_string ("video/x-h264, stream-format=(string)avc, "
      "codec_data=(buffer)0164001fffe1001c6764001facd9405005bb016a02020280000003008000001e478c18cb01000568ebecb22c");

  gst_harness_set_src_caps (v_src, gst_caps_ref (v_caps2));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (v_src,
          create_buffer (h264_buf, sizeof (h264_buf), base_time + duration,
              duration)));

  gst_harness_crank_single_clock_wait (mux);

  /* push from flvmux to demux */
  fail_unless_equals_int (GST_FLOW_OK, gst_harness_sink_push_many (mux, 3));

  /* should accept without the constraint */
  while ((event = gst_harness_try_pull_event (v_sink))) {
    if (GST_EVENT_TYPE (event) == GST_EVENT_CAPS) {
      gst_event_parse_caps (event, &ret_caps);
      GST_LOG ("v_caps2 %" GST_PTR_FORMAT ", ret caps %" GST_PTR_FORMAT,
          v_caps2, ret_caps);
      fail_unless (gst_caps_is_equal (v_caps2, ret_caps));
    }
    gst_event_unref (event);
  }

  /* verify we got 1x audio and 2x video buffers out of flvdemux */
  gst_buffer_unref (gst_harness_pull (a_sink));
  gst_buffer_unref (gst_harness_pull (v_sink));
  gst_buffer_unref (gst_harness_pull (v_sink));
  gst_caps_unref (a_caps1);
  gst_caps_unref (v_caps1);
  gst_caps_unref (v_caps2);

  gst_harness_teardown (a_src);
  gst_harness_teardown (v_src);
  gst_harness_teardown (mux);
  gst_harness_teardown (a_sink);
  gst_harness_teardown (v_sink);
}

GST_END_TEST;

GST_START_TEST (test_audio_caps_change_streamable)
{
  GstEvent *event;
  GstCaps *a_caps1, *a_caps2, *v_caps1, *ret_caps;
  GstHarness *mux = gst_harness_new_with_padnames ("flvmux", NULL, "src");
  GstHarness *a_src =
      gst_harness_new_with_element (mux->element, "audio", NULL);
  GstHarness *v_src =
      gst_harness_new_with_element (mux->element, "video", NULL);
  GstHarness *demux = gst_harness_new_with_padnames ("flvdemux", "sink", NULL);
  GstHarness *a_sink =
      gst_harness_new_with_element (demux->element, NULL, NULL);
  GstHarness *v_sink =
      gst_harness_new_with_element (demux->element, NULL, NULL);
  DemuxHarnesses harnesses = { a_sink, v_sink };

  const GstClockTime base_time = 123456789;
  const GstClockTime duration_ms = 20;
  const GstClockTime duration = duration_ms * GST_MSECOND;

  g_object_set (mux->element, "streamable", TRUE, NULL);

  g_signal_connect (demux->element, "pad-added",
      G_CALLBACK (flvdemux_pad_added), &harnesses);
  gst_harness_add_sink_harness (mux, demux);

  a_caps1 =
      gst_caps_from_string
      ("audio/mpeg, mpegversion=(int)4, framed=(boolean)true, stream-format=(string)raw, "
      "rate=(int)44100, channels=(int)1, codec_data=(buffer)1208");

  v_caps1 = gst_caps_from_string ("video/x-h264, stream-format=(string)avc, "
      "codec_data=(buffer)0142c00cffe1000b6742c00c95a7201e1108d401000468ce3c80");

  gst_harness_set_src_caps (a_src, gst_caps_ref (a_caps1));
  gst_harness_set_src_caps (v_src, gst_caps_ref (v_caps1));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (a_src,
          create_buffer (raw_frame_short, sizeof (raw_frame_short), base_time,
              duration)));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (v_src,
          create_buffer (h264_buf, sizeof (h264_buf), base_time, duration)));

  gst_harness_crank_single_clock_wait (mux);

  /* push from flvmux to demux */
  fail_unless_equals_int (GST_FLOW_OK, gst_harness_sink_push_many (mux, 6));

  /* should accept without the constraint */
  while ((event = gst_harness_try_pull_event (a_sink))) {
    if (GST_EVENT_TYPE (event) == GST_EVENT_CAPS) {
      gst_event_parse_caps (event, &ret_caps);
      GST_LOG ("a_caps1 %" GST_PTR_FORMAT ", ret caps %" GST_PTR_FORMAT,
          a_caps1, ret_caps);
      fail_unless (gst_caps_is_equal (a_caps1, ret_caps));
    }
    gst_event_unref (event);
  }

  /* caps change */
  a_caps2 =
      gst_caps_from_string
      ("audio/mpeg, mpegversion=(int)4, framed=(boolean)true, stream-format=(string)raw, "
      "rate=(int)22050, channels=(int)1, codec_data=(buffer)1388");

  gst_harness_set_src_caps (a_src, gst_caps_ref (a_caps2));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (a_src,
          create_buffer (raw_frame_short, sizeof (raw_frame_short),
              base_time + duration, duration)));

  gst_harness_crank_single_clock_wait (mux);

  /* push from flvmux to demux */
  fail_unless_equals_int (GST_FLOW_OK, gst_harness_sink_push_many (mux, 3));

  /* should accept without the constraint */
  while ((event = gst_harness_try_pull_event (a_sink))) {
    if (GST_EVENT_TYPE (event) == GST_EVENT_CAPS) {
      gst_event_parse_caps (event, &ret_caps);
      GST_LOG ("a_caps2 %" GST_PTR_FORMAT ", ret caps %" GST_PTR_FORMAT,
          a_caps2, ret_caps);
      fail_unless (gst_caps_is_equal (a_caps2, ret_caps));
    }
    gst_event_unref (event);
  }

  /* verify we got 2x audio and 1x video buffers out of flvdemux */
  gst_buffer_unref (gst_harness_pull (a_sink));
  gst_buffer_unref (gst_harness_pull (a_sink));
  gst_buffer_unref (gst_harness_pull (v_sink));
  gst_caps_unref (a_caps1);
  gst_caps_unref (a_caps2);
  gst_caps_unref (v_caps1);

  gst_harness_teardown (a_src);
  gst_harness_teardown (v_src);
  gst_harness_teardown (mux);
  gst_harness_teardown (a_sink);
  gst_harness_teardown (v_sink);
}

GST_END_TEST;

GST_START_TEST (test_video_caps_change_streamable_single)
{
  GstEvent *event;
  GstCaps *v_caps1, *v_caps2, *ret_caps;
  GstHarness *mux = gst_harness_new_with_padnames ("flvmux", NULL, "src");
  GstHarness *v_src =
      gst_harness_new_with_element (mux->element, "video", NULL);
  GstHarness *demux = gst_harness_new_with_padnames ("flvdemux", "sink", NULL);
  GstHarness *v_sink =
      gst_harness_new_with_element (demux->element, NULL, NULL);
  DemuxHarnesses harnesses = { NULL, v_sink };

  const GstClockTime base_time = 123456789;
  const GstClockTime duration_ms = 20;
  const GstClockTime duration = duration_ms * GST_MSECOND;

  g_object_set (mux->element, "streamable", TRUE, NULL);

  g_signal_connect (demux->element, "pad-added",
      G_CALLBACK (flvdemux_pad_added), &harnesses);
  gst_harness_add_sink_harness (mux, demux);

  v_caps1 = gst_caps_from_string ("video/x-h264, stream-format=(string)avc, "
      "codec_data=(buffer)0142c00cffe1000b6742c00c95a7201e1108d401000468ce3c80");

  gst_harness_set_src_caps (v_src, gst_caps_ref (v_caps1));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (v_src,
          create_buffer (h264_buf, sizeof (h264_buf), base_time, duration)));

  /* push from flvmux to demux */
  fail_unless_equals_int (GST_FLOW_OK, gst_harness_sink_push_many (mux, 4));

  /* should accept without the constraint */
  while ((event = gst_harness_try_pull_event (v_sink))) {
    if (GST_EVENT_TYPE (event) == GST_EVENT_CAPS) {
      gst_event_parse_caps (event, &ret_caps);
      GST_LOG ("v_caps1 %" GST_PTR_FORMAT ", ret caps %" GST_PTR_FORMAT,
          v_caps1, ret_caps);
      fail_unless (gst_caps_is_equal (v_caps1, ret_caps));
    }
    gst_event_unref (event);
  }

  /* caps change */
  v_caps2 = gst_caps_from_string ("video/x-h264, stream-format=(string)avc, "
      "codec_data=(buffer)0164001fffe1001c6764001facd9405005bb016a02020280000003008000001e478c18cb01000568ebecb22c");

  gst_harness_set_src_caps (v_src, gst_caps_ref (v_caps2));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (v_src,
          create_buffer (h264_buf, sizeof (h264_buf), base_time + duration,
              duration)));

  /* push from flvmux to demux */
  fail_unless_equals_int (GST_FLOW_OK, gst_harness_sink_push_many (mux, 3));

  /* should accept without the constraint */
  while ((event = gst_harness_try_pull_event (v_sink))) {
    if (GST_EVENT_TYPE (event) == GST_EVENT_CAPS) {
      gst_event_parse_caps (event, &ret_caps);
      GST_LOG ("v_caps2 %" GST_PTR_FORMAT ", ret caps %" GST_PTR_FORMAT,
          v_caps2, ret_caps);
      fail_unless (gst_caps_is_equal (v_caps2, ret_caps));
    }
    gst_event_unref (event);
  }

  /* verify we got 2x video buffers out of flvdemux */
  gst_buffer_unref (gst_harness_pull (v_sink));
  gst_buffer_unref (gst_harness_pull (v_sink));
  gst_caps_unref (v_caps1);
  gst_caps_unref (v_caps2);

  gst_harness_teardown (v_src);
  gst_harness_teardown (mux);
  gst_harness_teardown (v_sink);
}

GST_END_TEST;

GST_START_TEST (test_audio_caps_change_streamable_single)
{
  GstEvent *event;
  GstCaps *a_caps1, *a_caps2, *ret_caps;
  GstHarness *mux = gst_harness_new_with_padnames ("flvmux", NULL, "src");
  GstHarness *a_src =
      gst_harness_new_with_element (mux->element, "audio", NULL);
  GstHarness *demux = gst_harness_new_with_padnames ("flvdemux", "sink", NULL);
  GstHarness *a_sink =
      gst_harness_new_with_element (demux->element, NULL, NULL);
  DemuxHarnesses harnesses = { a_sink, NULL };

  const GstClockTime base_time = 123456789;
  const GstClockTime duration_ms = 20;
  const GstClockTime duration = duration_ms * GST_MSECOND;

  g_object_set (mux->element, "streamable", TRUE, NULL);

  g_signal_connect (demux->element, "pad-added",
      G_CALLBACK (flvdemux_pad_added), &harnesses);
  gst_harness_add_sink_harness (mux, demux);

  a_caps1 =
      gst_caps_from_string
      ("audio/mpeg, mpegversion=(int)4, framed=(boolean)true, stream-format=(string)raw, "
      "rate=(int)44100, channels=(int)1, codec_data=(buffer)1208");

  gst_harness_set_src_caps (a_src, gst_caps_ref (a_caps1));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (a_src,
          create_buffer (raw_frame_short, sizeof (raw_frame_short), base_time,
              duration)));

  /* push from flvmux to demux */
  fail_unless_equals_int (GST_FLOW_OK, gst_harness_sink_push_many (mux, 4));

  /* should accept without the constraint */
  while ((event = gst_harness_try_pull_event (a_sink))) {
    if (GST_EVENT_TYPE (event) == GST_EVENT_CAPS) {
      gst_event_parse_caps (event, &ret_caps);
      GST_LOG ("a_caps1 %" GST_PTR_FORMAT ", ret caps %" GST_PTR_FORMAT,
          a_caps1, ret_caps);
      fail_unless (gst_caps_is_equal (a_caps1, ret_caps));
    }
    gst_event_unref (event);
  }

  /* caps change */
  a_caps2 =
      gst_caps_from_string
      ("audio/mpeg, mpegversion=(int)4, framed=(boolean)true, stream-format=(string)raw, "
      "rate=(int)22050, channels=(int)1, codec_data=(buffer)1388");

  gst_harness_set_src_caps (a_src, gst_caps_ref (a_caps2));

  fail_unless_equals_int (GST_FLOW_OK, gst_harness_push (a_src,
          create_buffer (raw_frame_short, sizeof (raw_frame_short),
              base_time + duration, duration)));

  /* push from flvmux to demux */
  fail_unless_equals_int (GST_FLOW_OK, gst_harness_sink_push_many (mux, 3));

  /* should accept without the constraint */
  while ((event = gst_harness_try_pull_event (a_sink))) {
    if (GST_EVENT_TYPE (event) == GST_EVENT_CAPS) {
      gst_event_parse_caps (event, &ret_caps);
      GST_LOG ("a_caps2 %" GST_PTR_FORMAT ", ret caps %" GST_PTR_FORMAT,
          a_caps2, ret_caps);
      fail_unless (gst_caps_is_equal (a_caps2, ret_caps));
    }
    gst_event_unref (event);
  }

  /* verify we got 2x audio out of flvdemux */
  gst_buffer_unref (gst_harness_pull (a_sink));
  gst_buffer_unref (gst_harness_pull (a_sink));
  gst_caps_unref (a_caps1);
  gst_caps_unref (a_caps2);

  gst_harness_teardown (a_src);
  gst_harness_teardown (mux);
  gst_harness_teardown (a_sink);
}

GST_END_TEST;

typedef struct
{
  guint media_type;
  guint64 ts;                   /* timestamp in ms */
} InputData;

GST_START_TEST (test_incrementing_timestamps)
{
  GstPad *audio_sink, *video_sink, *audio_src, *video_src;
  GstHarness *h, *audio, *video, *audio_q, *video_q;
  guint i;
  GstEvent *event;
  guint32 prev_pts;
  InputData input[] = {
    {AUDIO, 155},
    {VIDEO, 156},
    {VIDEO, 190},
    {AUDIO, 176},
    {AUDIO, 197},
  };

  /* setup flvmuxer with queues in front */
  h = gst_harness_new_with_padnames ("flvmux", NULL, "src");
  audio = gst_harness_new_with_element (h->element, "audio", NULL);
  video = gst_harness_new_with_element (h->element, "video", NULL);
  audio_q = gst_harness_new ("queue");
  video_q = gst_harness_new ("queue");
  audio_sink = GST_PAD_PEER (audio->srcpad);
  video_sink = GST_PAD_PEER (video->srcpad);
  audio_src = GST_PAD_PEER (audio_q->sinkpad);
  video_src = GST_PAD_PEER (video_q->sinkpad);
  gst_pad_unlink (audio->srcpad, audio_sink);
  gst_pad_unlink (video->srcpad, video_sink);
  gst_pad_unlink (audio_src, audio_q->sinkpad);
  gst_pad_unlink (video_src, video_q->sinkpad);
  gst_pad_link (audio_src, audio_sink);
  gst_pad_link (video_src, video_sink);
  g_object_set (h->element, "streamable", TRUE, NULL);

  gst_harness_set_src_caps_str (audio_q,
      "audio/mpeg, mpegversion=(int)4, "
      "rate=(int)44100, channels=(int)1, "
      "stream-format=(string)raw, codec_data=(buffer)1208");

  gst_harness_set_src_caps_str (video_q,
      "video/x-h264, stream-format=(string)avc, alignment=(string)au, "
      "codec_data=(buffer)0142c00dffe1000d6742c00d95a0507c807844235001000468ce3c80");

  for (i = 0; i < G_N_ELEMENTS (input); i++) {
    InputData *d = &input[i];
    GstBuffer *buf = gst_buffer_new ();

    GST_BUFFER_DTS (buf) = GST_BUFFER_PTS (buf) = d->ts * GST_MSECOND;

    if (d->media_type == AUDIO)
      gst_harness_push (audio_q, buf);
    else
      gst_harness_push (video_q, buf);
  }

  gst_harness_push_event (audio_q, gst_event_new_eos ());
  gst_harness_push_event (video_q, gst_event_new_eos ());

  while ((event = gst_harness_pull_event (h)) != NULL) {
    GstEventType event_type = GST_EVENT_TYPE (event);
    gst_event_unref (event);

    if (event_type == GST_EVENT_EOS)
      break;
  }

  /* pull the flv metadata */
  gst_buffer_unref (gst_harness_pull (h));
  gst_buffer_unref (gst_harness_pull (h));
  gst_buffer_unref (gst_harness_pull (h));

  /* verify pts in the flvheader is increasing */
  prev_pts = 0;
  for (i = 0; i < G_N_ELEMENTS (input); i++) {
    GstBuffer *buf = gst_harness_pull (h);
    GstMapInfo map;
    guint32 pts;

    fail_unless (buf != NULL);

    gst_buffer_map (buf, &map, GST_MAP_READ);
    pts = GST_READ_UINT24_BE (map.data + 4);
    GST_DEBUG ("media=%u, pts = %u\n", map.data[0], pts);
    fail_unless (pts >= prev_pts);
    prev_pts = pts;
    gst_buffer_unmap (buf, &map);
    gst_buffer_unref (buf);
  }

  /* teardown */
  gst_harness_teardown (h);
  gst_harness_teardown (audio);
  gst_harness_teardown (video);
  gst_harness_teardown (audio_q);
  gst_harness_teardown (video_q);
}

GST_END_TEST;

GST_START_TEST (test_rollover_timestamps)
{
  GstPad *audio_sink, *video_sink, *audio_src, *video_src;
  GstHarness *h, *audio, *video, *audio_q, *video_q;
  GstEvent *event;
  guint i;
  guint64 rollover_pts = (guint64) G_MAXUINT32 + 100;
  InputData input[] = {
    {AUDIO, 0}
    ,
    {VIDEO, 0}
    ,
    {VIDEO, (guint64) G_MAXUINT32 - 100}
    ,
    {AUDIO, (guint64) G_MAXUINT32 - 95}
    ,
    {AUDIO, rollover_pts}
    ,
  };

  /* setup flvmuxer with queues in front */
  h = gst_harness_new_with_padnames ("flvmux", NULL, "src");
  audio = gst_harness_new_with_element (h->element, "audio", NULL);
  video = gst_harness_new_with_element (h->element, "video", NULL);
  audio_q = gst_harness_new ("queue");
  video_q = gst_harness_new ("queue");
  audio_sink = GST_PAD_PEER (audio->srcpad);
  video_sink = GST_PAD_PEER (video->srcpad);
  audio_src = GST_PAD_PEER (audio_q->sinkpad);
  video_src = GST_PAD_PEER (video_q->sinkpad);
  gst_pad_unlink (audio->srcpad, audio_sink);
  gst_pad_unlink (video->srcpad, video_sink);
  gst_pad_unlink (audio_src, audio_q->sinkpad);
  gst_pad_unlink (video_src, video_q->sinkpad);
  gst_pad_link (audio_src, audio_sink);
  gst_pad_link (video_src, video_sink);
  g_object_set (h->element, "streamable", TRUE, NULL);

  gst_harness_set_src_caps_str (audio_q,
      "audio/mpeg, mpegversion=(int)4, "
      "rate=(int)44100, channels=(int)1, "
      "stream-format=(string)raw, codec_data=(buffer)1208");

  gst_harness_set_src_caps_str (video_q,
      "video/x-h264, stream-format=(string)avc, alignment=(string)au, "
      "codec_data=(buffer)0142c00dffe1000d6742c00d95a0507c807844235001000468ce3c80");

  for (i = 0; i < G_N_ELEMENTS (input); i++) {
    InputData *d = &input[i];
    GstBuffer *buf = gst_buffer_new ();

    GST_BUFFER_DTS (buf) = GST_BUFFER_PTS (buf) = d->ts * GST_MSECOND;
    GST_DEBUG ("Push media=%u, pts=%" G_GUINT64_FORMAT " (%" GST_TIME_FORMAT
        ")", d->media_type, d->ts, GST_TIME_ARGS (GST_BUFFER_PTS (buf)));

    if (d->media_type == AUDIO)
      gst_harness_push (audio_q, buf);
    else
      gst_harness_push (video_q, buf);

  }
  gst_harness_push_event (audio_q, gst_event_new_eos ());
  gst_harness_push_event (video_q, gst_event_new_eos ());

  while ((event = gst_harness_pull_event (h)) != NULL) {
    GstEventType event_type = GST_EVENT_TYPE (event);
    gst_event_unref (event);

    if (event_type == GST_EVENT_EOS)
      break;
  }

  /* pull the flv metadata */
  gst_buffer_unref (gst_harness_pull (h));
  gst_buffer_unref (gst_harness_pull (h));
  gst_buffer_unref (gst_harness_pull (h));
  gst_buffer_unref (gst_harness_pull (h));

  /* verify rollover pts in the flvheader is handled */
  for (i = 0; i < G_N_ELEMENTS (input); i++) {
    GstBuffer *buf = gst_harness_pull (h);
    GstMapInfo map;
    guint32 pts, pts_ext;

    fail_unless (buf != NULL);

    gst_buffer_map (buf, &map, GST_MAP_READ);
    pts = GST_READ_UINT24_BE (map.data + 4);
    pts_ext = GST_READ_UINT8 (map.data + 7);
    pts |= pts_ext << 24;
    GST_DEBUG ("media=%u, pts=%u (%" GST_TIME_FORMAT ")",
        map.data[0], pts, GST_TIME_ARGS (pts * GST_MSECOND));
    fail_unless (pts == (guint32) input[i].ts);
    gst_buffer_unmap (buf, &map);
    gst_buffer_unref (buf);
  }

  /* teardown */
  gst_harness_teardown (h);
  gst_harness_teardown (audio);
  gst_harness_teardown (video);
  gst_harness_teardown (audio_q);
  gst_harness_teardown (video_q);
}

GST_END_TEST;

#define H264_FILE GST_TEST_FILES_PATH G_DIR_SEPARATOR_S "flv_test/h264/%d.h264"
#define H265_FILE GST_TEST_FILES_PATH G_DIR_SEPARATOR_S "flv_test/h265/%d.h265"
#define AAC_FILE GST_TEST_FILES_PATH G_DIR_SEPARATOR_S "flv_test/aac/%d.aac"
#define MP3_8K_FILE GST_TEST_FILES_PATH G_DIR_SEPARATOR_S "flv_test/mp3_8k/%d.mp3"
#define H264_CAPS "video/x-h264,stream-format=avc,codec_data=(buffer)01f4000dffe1001b67f4000d91968141fb016a0c0c0c80000003008000001e478a155001000468ce3192fff8f800,framerate=30/1"
#define AAC_CAPS "audio/mpeg,mpegversion=(int)4,channels=(int)1,framed=(boolean)true,rate=(int)44100,codec_data=(buffer)1208,stream-format=(string)raw"
#define MP3_8K_CAPS "audio/mpeg, mpegversion=(int)1, layer=(int)3, rate=(int)8000, channels=(int)2, parsed=(boolean)true"
#define H265_CAPS "video/x-h265, stream-format=(string)hvc1, alignment=(string)au, pixel-aspect-ratio=(fraction)1/1, framerate=(fraction)30/1, \
                codec_data=(buffer)0104080000009e08000000003ff000fcfff8f800000f03200001001740010c01ffff0408000003009e0800000300003f959809210001\
002f4201010408000003009e0800000300003f90014101e2cb2b3492657ff80008000b506060604000000300400000078222000100074401c172b46240"

typedef struct
{
  GstClockTime current_ts;
  GAsyncQueue *in_buffer_queue;
  GstCaps *caps;
  GstPad *input_src_pad;
  GstPad *demuxer_src_pad;
  gulong input_src_pad_probe_id;
  gulong demuxer_src_pad_probe_id;
} TrackDetails;

typedef struct
{
  guint num_audio_tracks;
  guint num_video_tracks;
  GstElement *pipeline;
  TrackDetails *audio_tracks;
  TrackDetails *video_tracks;
} TracksInfo;

static GstPadProbeReturn
input_src_pad_probe_callback (GstPad * pad, GstPadProbeInfo * info,
    TrackDetails * track)
{
  if (info->type & GST_PAD_PROBE_TYPE_BUFFER) {
    GstBuffer *buffer = info->data;
    fail_unless (track != NULL);

    GstClockTime ts = track->current_ts;
    guint duration = 33 * GST_MSECOND;
    GST_BUFFER_DURATION (buffer) = duration;
    GST_BUFFER_DTS (buffer) = GST_BUFFER_PTS (buffer) = ts;
    GST_TRACE ("probe type GST_PAD_PROBE_TYPE_BUFFER %" GST_PTR_FORMAT, buffer);

    // store buffers for comparing with demuxer output
    gst_buffer_ref (buffer);
    g_async_queue_push (track->in_buffer_queue, buffer);

    track->current_ts = ts + duration;
  }

  if (info->type & GST_PAD_PROBE_TYPE_EVENT_DOWNSTREAM) {
    GstEvent *event = GST_PAD_PROBE_INFO_EVENT (info);
    switch (GST_EVENT_TYPE (event)) {
      case GST_EVENT_SEGMENT:{
        GstSegment segment_new;
        GST_TRACE ("gst segment event %" GST_PTR_FORMAT, event);
        gst_segment_init (&segment_new, GST_FORMAT_TIME);
        guint32 seqnum = gst_event_get_seqnum (event);
        GstEvent *event_new = gst_event_new_segment (&segment_new);
        gst_event_unref (info->data);
        info->data = event_new;

        gst_event_set_seqnum (event_new, seqnum);
        GST_TRACE ("gst segment event new %" GST_PTR_FORMAT, event_new);
      }
        break;
      default:
        break;
    }
  }
  return GST_PAD_PROBE_OK;
}

static GstPadProbeReturn
demux_src_pad_probe_callback (GstPad * pad, GstPadProbeInfo * info,
    GAsyncQueue * buffer_queue)
{
  if (info->type & GST_PAD_PROBE_TYPE_BUFFER) {
    GstBuffer *buffer = info->data;
    GstMapInfo map = GST_MAP_INFO_INIT;
    GstBuffer *in_buffer = NULL;
    GstMapInfo in_buf_map = GST_MAP_INFO_INIT;

    GST_TRACE ("demuxer src pad: %" GST_PTR_FORMAT, buffer);

    in_buffer = g_async_queue_try_pop (buffer_queue);

    fail_unless (in_buffer != NULL);
    fail_unless_equals_uint64 (GST_BUFFER_PTS (buffer),
        GST_BUFFER_PTS (in_buffer));

    gst_buffer_map (buffer, &map, GST_MAP_READ);
    gst_buffer_map (in_buffer, &in_buf_map, GST_MAP_READ);

    fail_unless_equals_int (map.size, in_buf_map.size);
    fail_unless_equals_int (0, memcmp (map.data, in_buf_map.data, map.size));

    gst_buffer_unmap (buffer, &map);
    gst_buffer_unmap (in_buffer, &in_buf_map);
    gst_buffer_unref (in_buffer);
  }

  return GST_PAD_PROBE_OK;
}

static void
demux_multitrack_pad_added_handler (GstElement * flvdemux, GstPad * srcpad,
    TracksInfo * tracks_info)
{
  GstCaps *caps = gst_pad_get_current_caps (srcpad);
  gint id = 0;
  gchar *endptr;
  GstElement *fakesink = NULL;
  GstPad *fake_sink_pad = NULL;
  gchar *src_pad_name = GST_PAD_NAME (srcpad);
  // pad index is track id, so postition is after audio_ or video_
  guint8 track_id_pos = 6;
  TrackDetails *this_track = NULL;

  GST_TRACE ("pad %s added", src_pad_name);

  if (g_str_has_prefix (src_pad_name, "audio_")) {
    id = g_ascii_strtoll (src_pad_name + track_id_pos, &endptr, 10);
    if (endptr != src_pad_name + track_id_pos && id > 0 && id < G_MAXUINT8
        && id <= tracks_info->num_audio_tracks) {
      this_track = &tracks_info->audio_tracks[id];
      fail_unless (gst_caps_is_equal (caps, this_track->caps));
    } else {
      fail ("Invalid audio track id %s", src_pad_name + track_id_pos);
    }
  } else if (g_str_has_prefix (src_pad_name, "video_")) {
    id = g_ascii_strtoll (src_pad_name + track_id_pos, &endptr, 10);
    if (endptr != src_pad_name + track_id_pos && id > 0 && id < G_MAXUINT8
        && id <= tracks_info->num_video_tracks) {
      this_track = &tracks_info->video_tracks[id];
      fail_unless (gst_caps_is_equal (caps, this_track->caps));
    } else {
      fail ("Invalid video track id %s", src_pad_name + track_id_pos);
    }
  } else if (g_str_equal ("audio", src_pad_name)) {
    this_track = &tracks_info->audio_tracks[id];
    fail_unless (gst_caps_is_equal (caps, this_track->caps));
    GST_TRACE ("audio pad caps matched");
  } else if (g_str_equal ("video", src_pad_name)) {
    this_track = &tracks_info->video_tracks[id];
    fail_unless (gst_caps_is_equal (caps, this_track->caps));
  } else {
    ck_abort_msg ("Unexpected demux pad: %s", GST_STR_NULL (src_pad_name));
  }

  fail_unless (this_track->demuxer_src_pad == NULL);
  this_track->demuxer_src_pad = srcpad;

  this_track->demuxer_src_pad_probe_id =
      gst_pad_add_probe (srcpad, GST_PAD_PROBE_TYPE_BUFFER,
      (GstPadProbeCallback) demux_src_pad_probe_callback,
      this_track->in_buffer_queue, NULL);

  fakesink = gst_element_factory_make ("fakesink", NULL);
  fail_unless (fakesink != NULL);
  g_object_set (fakesink, "async", FALSE, "sync", FALSE, NULL);

  fail_unless (gst_bin_add (GST_BIN (tracks_info->pipeline), fakesink));

  gst_element_sync_state_with_parent (fakesink);
  fake_sink_pad = gst_element_get_static_pad (fakesink, "sink");
  fail_unless_equals_int (gst_pad_link (srcpad, fake_sink_pad),
      GST_PAD_LINK_OK);

  gst_object_unref (fake_sink_pad);
  gst_caps_unref (caps);
}

static void
run_pipeline (const gchar * pipeline_str, TracksInfo * tracks_info)
{
  GstElement *pipeline;
  GstElement *muxer;
  fail_if (pipeline_str == NULL);
  fail_if (tracks_info == NULL);

  pipeline = gst_parse_launch (pipeline_str, NULL);
  fail_unless (pipeline != NULL);
  tracks_info->pipeline = pipeline;

  muxer = gst_bin_get_by_name (GST_BIN (pipeline), "mux");
  fail_unless (muxer != NULL);

  GST_OBJECT_LOCK (muxer);
  for (GList * p = muxer->sinkpads; p; p = p->next) {
    GstPad *mux_pad = p->data;
    GstPad *peer_src_pad = gst_pad_get_peer (mux_pad);
    fail_unless (peer_src_pad != NULL);

    gchar *mux_pad_name = GST_PAD_NAME (mux_pad);
    // pad index is track id, so postition is after audio_ or video_
    guint8 track_id_pos = 6;
    gchar *endptr = NULL;
    guint8 id = 0;              // track id
    TrackDetails *this_track = NULL;

    GST_TRACE ("mux pad %s peer_pad %s", mux_pad_name,
        GST_PAD_NAME (peer_src_pad));

    if (g_str_has_prefix (mux_pad_name, "audio_")) {
      fail_if (tracks_info->audio_tracks == NULL);
      id = g_ascii_strtoll (mux_pad_name + track_id_pos, &endptr, 10);

      // audio_0 not allowed, the element should error out
      fail_if (id == 0);
      this_track = &tracks_info->audio_tracks[id];

      fail_unless (endptr != mux_pad_name + track_id_pos
          && id <= tracks_info->num_audio_tracks);
    } else if (g_str_has_prefix (mux_pad_name, "video_")) {
      fail_if (tracks_info->video_tracks == NULL);
      id = g_ascii_strtoll (mux_pad_name + track_id_pos, &endptr, 10);

      // video_0 not allowed, the element should error out
      fail_if (id == 0);
      this_track = &tracks_info->video_tracks[id];

      fail_unless (endptr != mux_pad_name + track_id_pos
          && id <= tracks_info->num_video_tracks);
    } else if (g_str_equal ("audio", mux_pad_name)) {

      this_track = &tracks_info->audio_tracks[0];
    } else if (g_str_equal ("video", mux_pad_name)) {
      this_track = &tracks_info->video_tracks[0];
    }

    this_track->input_src_pad = peer_src_pad;
    this_track->input_src_pad_probe_id =
        gst_pad_add_probe (peer_src_pad,
        GST_PAD_PROBE_TYPE_BUFFER | GST_PAD_PROBE_TYPE_EVENT_DOWNSTREAM,
        (GstPadProbeCallback) input_src_pad_probe_callback, this_track, NULL);
  }
  GST_OBJECT_UNLOCK (muxer);
  gst_object_unref (muxer);

  GstElement *demux = gst_bin_get_by_name (GST_BIN (pipeline), "demux");
  fail_unless (demux != NULL);

  g_signal_connect (demux, "pad-added",
      G_CALLBACK (demux_multitrack_pad_added_handler), tracks_info);

  fail_unless_equals_int (gst_element_set_state (pipeline, GST_STATE_PLAYING),
      GST_STATE_CHANGE_SUCCESS);

  GstBus *bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));
  gboolean running = TRUE;
  GstMessage *message;

  while (running) {
    message = gst_bus_timed_pop_filtered (bus, -1, GST_MESSAGE_ANY);

    g_assert (message != NULL);

    switch (message->type) {
      case GST_MESSAGE_EOS:
        GST_TRACE ("got EOS");
        running = FALSE;
        fail_unless_equals_int (demux->numsrcpads,
            tracks_info->num_audio_tracks + tracks_info->num_video_tracks);
        break;
      case GST_MESSAGE_WARNING:{
        GError *gerror;
        gchar *debug;

        gst_message_parse_warning (message, &gerror, &debug);
        gst_object_default_error (GST_MESSAGE_SRC (message), gerror, debug);
        fail (debug);
      }
      case GST_MESSAGE_ERROR:
      {
        GError *gerror;
        gchar *debug;

        gst_message_parse_error (message, &gerror, &debug);
        gst_object_default_error (GST_MESSAGE_SRC (message), gerror, debug);
        g_error_free (gerror);
        g_free (debug);
        running = FALSE;
        break;
      }
      default:
        break;
    }
    gst_message_unref (message);
  }
  gst_object_unref (bus);

  for (guint8 t = 0; t < tracks_info->num_audio_tracks; t++) {
    TrackDetails *this_track = &tracks_info->audio_tracks[t];
    gst_caps_unref (this_track->caps);

    gst_pad_remove_probe (this_track->input_src_pad,
        this_track->input_src_pad_probe_id);
    gst_pad_remove_probe (this_track->demuxer_src_pad,
        this_track->demuxer_src_pad_probe_id);

    fail_unless_equals_int (g_async_queue_length (this_track->in_buffer_queue)
        , 0);
    g_async_queue_unref (this_track->in_buffer_queue);
    gst_object_unref (this_track->input_src_pad);
  }

  for (guint8 t = 0; t < tracks_info->num_video_tracks; t++) {
    TrackDetails *this_track = &tracks_info->video_tracks[t];
    gst_caps_unref (this_track->caps);

    gst_pad_remove_probe (this_track->input_src_pad,
        this_track->input_src_pad_probe_id);
    gst_pad_remove_probe (this_track->demuxer_src_pad,
        this_track->demuxer_src_pad_probe_id);

    fail_unless_equals_int (g_async_queue_length (this_track->in_buffer_queue)
        , 0);
    g_async_queue_unref (this_track->in_buffer_queue);
    gst_object_unref (this_track->input_src_pad);
  }

  gst_object_unref (demux);
  gst_element_set_state (pipeline, GST_STATE_NULL);

  gst_object_unref (pipeline);
}

GST_START_TEST (test_multiple_audio_and_video_tracks)
{
  enum
  {
    AAC_TRACK_ID = 0,
    MP3_TRACK_ID = 1,
    MAX_AUDIO_TRACKS,
  };

  enum
  {
    H264_TRACK_ID = 0,
    H265_TRACK_ID = 1,
    MAX_VIDEO_TRACKS,
  };

  TrackDetails video_tracks[MAX_VIDEO_TRACKS] = { 0, };
  TrackDetails audio_tracks[MAX_AUDIO_TRACKS] = { 0, };
  TracksInfo tracks_info = { 0, };

  const gchar *pipeline_str =
      "multifilesrc location=\"" AAC_FILE "\" caps=\"" AAC_CAPS "\" ! mux.audio \
     multifilesrc location=\"" MP3_8K_FILE "\" caps=\"" MP3_8K_CAPS
      "\" ! mux.audio_1 \
     multifilesrc location=\"" H264_FILE "\" caps=\"" H264_CAPS "\" ! mux.video \
     multifilesrc location=\"" H265_FILE "\" caps=\"" H265_CAPS "\" ! mux.video_1 \
     eflvmux name=mux ! flvdemux name=demux  \
    ";

  audio_tracks[AAC_TRACK_ID].caps = gst_caps_from_string (AAC_CAPS);
  audio_tracks[AAC_TRACK_ID].current_ts = 0;
  audio_tracks[AAC_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  audio_tracks[MP3_TRACK_ID].caps = gst_caps_from_string (MP3_8K_CAPS);
  audio_tracks[MP3_TRACK_ID].current_ts = 0;
  audio_tracks[MP3_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  tracks_info.audio_tracks = audio_tracks;
  tracks_info.num_audio_tracks = MAX_AUDIO_TRACKS;
  tracks_info.pipeline = NULL;

  video_tracks[H264_TRACK_ID].caps = gst_caps_from_string (H264_CAPS);
  video_tracks[H264_TRACK_ID].current_ts = 0;
  video_tracks[H264_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  video_tracks[H265_TRACK_ID].caps = gst_caps_from_string (H265_CAPS);
  video_tracks[H265_TRACK_ID].current_ts = 0;
  video_tracks[H265_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  tracks_info.video_tracks = video_tracks;
  tracks_info.num_video_tracks = MAX_VIDEO_TRACKS;
  tracks_info.pipeline = NULL;

  run_pipeline (pipeline_str, &tracks_info);
}

GST_END_TEST;


GST_START_TEST (test_multiple_audio_tracks)
{
  enum
  {
    AAC_TRACK_ID = 0,
    MP3_TRACK_ID = 1,
    MAX_TRACKS,
  };

  TrackDetails audio_tracks[MAX_TRACKS] = { 0, };
  TracksInfo audio_tracks_info = { 0, };

  const gchar *pipeline_str =
      "multifilesrc location=\"" AAC_FILE "\" caps=\"" AAC_CAPS "\" ! mux.audio \
     multifilesrc location=\"" MP3_8K_FILE "\" caps=\"" MP3_8K_CAPS
      "\" ! mux.audio_1 \
     eflvmux name=mux ! flvdemux name=demux  \
    ";

  audio_tracks[AAC_TRACK_ID].caps = gst_caps_from_string (AAC_CAPS);
  audio_tracks[AAC_TRACK_ID].current_ts = 0;
  audio_tracks[AAC_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  audio_tracks[MP3_TRACK_ID].caps = gst_caps_from_string (MP3_8K_CAPS);
  audio_tracks[MP3_TRACK_ID].current_ts = 0;
  audio_tracks[MP3_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  audio_tracks_info.audio_tracks = audio_tracks;
  audio_tracks_info.num_audio_tracks = MAX_TRACKS;
  audio_tracks_info.pipeline = NULL;

  run_pipeline (pipeline_str, &audio_tracks_info);
}

GST_END_TEST
GST_START_TEST (test_multiple_video_tracks)
{
  enum
  {
    H264_TRACK_ID = 0,
    H265_TRACK_ID = 1,
    MAX_TRACKS,
  };

  TrackDetails video_tracks[MAX_TRACKS] = { 0, };
  TracksInfo video_tracks_info = { 0, };

  const gchar *pipeline_str =
      "multifilesrc location=\"" H264_FILE "\" caps=\"" H264_CAPS "\" ! mux.video \
     multifilesrc location=\"" H265_FILE "\" caps=\"" H265_CAPS
      "\" ! mux.video_1 \
     eflvmux name=mux ! flvdemux name=demux \
    ";

  video_tracks[H264_TRACK_ID].caps = gst_caps_from_string (H264_CAPS);
  video_tracks[H264_TRACK_ID].current_ts = 0;
  video_tracks[H264_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  video_tracks[H265_TRACK_ID].caps = gst_caps_from_string (H265_CAPS);
  video_tracks[H265_TRACK_ID].current_ts = 0;
  video_tracks[H265_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  video_tracks_info.video_tracks = video_tracks;
  video_tracks_info.num_video_tracks = MAX_TRACKS;
  video_tracks_info.pipeline = NULL;

  run_pipeline (pipeline_str, &video_tracks_info);
}

GST_END_TEST;

GST_START_TEST (test_default_track_eflv_non_multitrack)
{
  enum
  {
    H264_TRACK_ID = 0,
    MAX_TRACKS,
  };

  TrackDetails video_tracks[MAX_TRACKS] = { 0, };
  TracksInfo video_tracks_info = { 0, };

  const gchar *pipeline_str =
      "multifilesrc location=\"" H264_FILE "\" caps=\"" H264_CAPS
      "\" ! mux.video \
     eflvmux name=mux video::flv-track-mode=non-multitrack ! flvdemux name=demux \
    ";

  video_tracks[H264_TRACK_ID].caps = gst_caps_from_string (H264_CAPS);
  video_tracks[H264_TRACK_ID].current_ts = 0;
  video_tracks[H264_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  video_tracks_info.video_tracks = video_tracks;
  video_tracks_info.num_video_tracks = MAX_TRACKS;
  video_tracks_info.pipeline = NULL;

  run_pipeline (pipeline_str, &video_tracks_info);
}

GST_END_TEST;


GST_START_TEST (test_default_track_eflv_multitrack_single_audio_single_video)
{
  enum
  {
    AAC_TRACK_ID = 0,
    MAX_AUDIO_TRACKS,
  };
  enum
  {
    H264_TRACK_ID = 0,
    MAX_VIDEO_TRACKS,
  };


  TrackDetails audio_tracks[MAX_AUDIO_TRACKS] = { 0, };
  TrackDetails video_tracks[MAX_VIDEO_TRACKS] = { 0, };
  TracksInfo tracks_info = { 0, };

  const gchar *pipeline_str =
      "multifilesrc location=\"" AAC_FILE "\" caps=\"" AAC_CAPS
      "\" ! mux.audio \
      multifilesrc location=\"" H264_FILE "\" caps=\"" H264_CAPS "\" ! mux.video \
     eflvmux name=mux audio::flv-track-mode=multitrack ! flvdemux name=demux  \
    ";

  audio_tracks[AAC_TRACK_ID].caps = gst_caps_from_string (AAC_CAPS);
  audio_tracks[AAC_TRACK_ID].current_ts = 0;
  audio_tracks[AAC_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  tracks_info.audio_tracks = audio_tracks;
  tracks_info.num_audio_tracks = MAX_AUDIO_TRACKS;
  tracks_info.pipeline = NULL;

  video_tracks[H264_TRACK_ID].caps = gst_caps_from_string (H264_CAPS);
  video_tracks[H264_TRACK_ID].current_ts = 0;
  video_tracks[H264_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  tracks_info.video_tracks = video_tracks;
  tracks_info.num_video_tracks = MAX_VIDEO_TRACKS;
  tracks_info.pipeline = NULL;

  run_pipeline (pipeline_str, &tracks_info);
}

GST_END_TEST;

GST_START_TEST (test_default_track_eflv_multitrack)
{
  enum
  {
    AAC_TRACK_ID = 0,
    MAX_TRACKS,
  };

  TrackDetails audio_tracks[MAX_TRACKS] = { 0, };
  TracksInfo audio_tracks_info = { 0, };

  const gchar *pipeline_str =
      "multifilesrc location=\"" AAC_FILE "\" caps=\"" AAC_CAPS
      "\" ! mux.audio \
     eflvmux name=mux audio::flv-track-mode=multitrack ! flvdemux name=demux  \
    ";

  audio_tracks[AAC_TRACK_ID].caps = gst_caps_from_string (AAC_CAPS);
  audio_tracks[AAC_TRACK_ID].current_ts = 0;
  audio_tracks[AAC_TRACK_ID].in_buffer_queue = g_async_queue_new ();

  audio_tracks_info.audio_tracks = audio_tracks;
  audio_tracks_info.num_audio_tracks = MAX_TRACKS;
  audio_tracks_info.pipeline = NULL;

  run_pipeline (pipeline_str, &audio_tracks_info);
}

GST_END_TEST;

GST_START_TEST (test_eflv_reject_pad_0)
{
  GstElement *eflvmux = gst_element_factory_make ("eflvmux", NULL);
  GstPad *audio_0_pad = gst_element_request_pad_simple (eflvmux, "audio_0");

  fail_unless (audio_0_pad == NULL);
  gst_object_unref (eflvmux);
}

GST_END_TEST;

static Suite *
flvmux_suite (void)
{
  Suite *s = suite_create ("flvmux");
  TCase *tc_chain = tcase_create ("general");
  gint loop = 16;

  suite_add_tcase (s, tc_chain);

#ifdef HAVE_VALGRIND
  if (RUNNING_ON_VALGRIND) {
    loop = 1;
  }
#endif

  tcase_add_loop_test (tc_chain, test_index_writing, 0, loop);

  tcase_add_test (tc_chain, test_speex_streamable);
  tcase_add_test (tc_chain, test_increasing_timestamp_when_pts_none);
  tcase_add_test (tc_chain, test_video_caps_late);
  tcase_add_test (tc_chain, test_audio_caps_change_streamable);
  tcase_add_test (tc_chain, test_video_caps_change_streamable);
  tcase_add_test (tc_chain, test_audio_caps_change_streamable_single);
  tcase_add_test (tc_chain, test_video_caps_change_streamable_single);
  tcase_add_test (tc_chain, test_incrementing_timestamps);
  tcase_add_test (tc_chain, test_rollover_timestamps);
  tcase_add_test (tc_chain, test_multiple_audio_tracks);
  tcase_add_test (tc_chain, test_multiple_video_tracks);
  tcase_add_test (tc_chain, test_default_track_eflv_multitrack);
  tcase_add_test (tc_chain, test_default_track_eflv_non_multitrack);
  tcase_add_test (tc_chain, test_multiple_audio_and_video_tracks);
  tcase_add_test (tc_chain,
      test_default_track_eflv_multitrack_single_audio_single_video);
  tcase_add_test (tc_chain, test_eflv_reject_pad_0);

  return s;
}

GST_CHECK_MAIN (flvmux)
