if(NOT DEFINED ENV{JENKINS_HOME})
	if(NOT DEFINED ENV{CI_API_V4_URL})
		set(ENV{CI_API_V4_URL} "https://gitlab.govkg.de/api/v4")
	endif()
	if(NOT DEFINED ENV{CI_PROJECT_ID})
		set(ENV{CI_PROJECT_ID} 786)
	endif()

	if(DEFINED ENV{CI_JOB_TOKEN})
		set(ENV{TOKEN_TYPE} "JOB-TOKEN")
		set(ENV{TOKEN} $ENV{CI_JOB_TOKEN})
	elseif(DEFINED ENV{CI_PIPELINE_TOKEN})
		set(ENV{TOKEN_TYPE} "PRIVATE-TOKEN")
		set(ENV{TOKEN} $ENV{CI_PIPELINE_TOKEN})
	else()
		message(FATAL_ERROR "No CI_JOB_TOKEN / CI_PIPELINE_TOKEN found")
	endif()
endif()

function(handle_http _file)
	set(options UPLOAD ALLOW_404)
	set(multiValueArgs URLS HEADER)
	cmake_parse_arguments(_PARAM "${options}" "${oneValueArgs}" "${multiValueArgs}" ${ARGN})

	if(_PARAM_HEADER)
		foreach(_header ${_PARAM_HEADER})
			list(APPEND HEADER HTTPHEADER "${_header}")
		endforeach()
	else()
		list(APPEND HEADER HTTPHEADER "$ENV{TOKEN_TYPE}: $ENV{TOKEN}")
	endif()

	foreach(url ${_PARAM_URLS})
		if(_PARAM_UPLOAD)
			set(operation "Upload")
			message(STATUS "Upload file: ${url}")
			if(NOT ECHO)
				if(NOT DEFINED ENV{CI})
					message(FATAL_ERROR "No CI detected: skip upload!")
					return()
				endif()
				file(UPLOAD "${_file}" "${url}" ${HEADER} SHOW_PROGRESS STATUS status LOG output)
			endif()
		else()
			set(operation "Download")
			message(STATUS "Download file: ${url}")
			if(NOT ECHO)
				file(DOWNLOAD "${url}" "${_file}" ${HEADER} SHOW_PROGRESS STATUS status LOG output)
			endif()
		endif()

		list(GET status 0 status_code)
		if(status_code EQUAL 0)
			message(STATUS "${operation} succeeded: ${_file}")
			return()
		else()
			message(STATUS "${operation} failed: ${_file}")
			if(NOT _PARAM_UPLOAD)
				step(${CMAKE_COMMAND} -E rm ${_file})
			endif()

			if(NOT ECHO)
				string(REGEX REPLACE "\\[[0-9]+ bytes data\\]" "" output "${output}")
				string(REGEX REPLACE "$ENV{TOKEN}" "***" output "${output}")
				string(REGEX MATCH "HTTP/[0-9.]+ 404" http_404 "${output}")
				if(http_404 AND _PARAM_ALLOW_404)
					continue()
				endif()
				message(FATAL_ERROR ${output})
			endif()
		endif()
	endforeach()
endfunction()

function(handle_pkg _file _dirs _version)
	foreach(dir ${_dirs})
		cmake_path(GET _file FILENAME pkg)
		list(APPEND URLS "$ENV{CI_API_V4_URL}/projects/$ENV{CI_PROJECT_ID}/packages/generic/${dir}/${_version}/${pkg}")
	endforeach()

	handle_http("${_file}" ${ARGN} ALLOW_404 URLS ${URLS})
endfunction()

function(handle_pkg_libs _file _dirs)
	foreach(dir IN LISTS _dirs)
		list(APPEND suffixedDirs "${dir}-OpenSSL_${LIBS_OPENSSL}-Qt_${LIBS_QT}")
	endforeach()
	handle_pkg("${_file}" "${suffixedDirs}" "${LIBS_IDENTIFIER}" ${ARGN})
endfunction()

function(download_pkg_libs _file)
	list(APPEND dir ${LIBS_BRANCH})
	if(REVIEW AND LIBS_PHASE)
		list(APPEND dir ${LIBS_BRANCH}-${LIBS_PHASE})
	endif()
	handle_pkg_libs("${_file}" "${dir}")
endfunction()

function(upload_pkg_libs _file)
	set(dir ${LIBS_BRANCH})
	if(LIBS_PHASE)
		set(dir ${dir}-${LIBS_PHASE})
	endif()
	handle_pkg_libs("${_file}" "${dir}" UPLOAD)
endfunction()

function(download_file _file _id)
	list(APPEND URLS "$ENV{CI_API_V4_URL}/projects/$ENV{CI_PROJECT_ID}/secure_files/${_id}/download")
	handle_http("${_file}" URLS ${URLS})
endfunction()

function(prepare_android_keystore _store _prefix)
	if(NOT DEFINED ENV{GITLAB_CI})
		return()
	endif()

	string(TOLOWER "${_store}" filename)
	set(store "$ENV{HOME}/.android/${filename}.keystore")
	set(prefixEnv "ANDROID_KEYSTORE_${_store}")
	set(file_id "${prefixEnv}_FILE_ID")
	set(alias "${prefixEnv}_ALIAS")
	set(psw "${prefixEnv}_PSW")

	if(NOT EXISTS "${store}")
		if(NOT DEFINED ENV{${file_id}})
			message(FATAL_ERROR "Missing variable: ${file_id}")
		endif()
		download_file("${store}" $ENV{${file_id}})
		if(NOT EXISTS "${store}")
			message(FATAL_ERROR "Cannot fetch file: ${store}")
		endif()
	endif()

	set("${_prefix}" "${store}" PARENT_SCOPE)
	set("${_prefix}_ALIAS" "$ENV{${alias}}" PARENT_SCOPE)
	set("${_prefix}_PSW" "$ENV{${psw}}" PARENT_SCOPE)
endfunction()

function(prepare_gpg _var)
	set(options SH)
	cmake_parse_arguments(_PARAM "${options}" "${oneValueArgs}" "${multiValueArgs}" ${ARGN})

	set(GPG_CMD gpg --batch --passphrase $ENV{GPG_PSW} --pinentry-mode loopback)
	if(_PARAM_SH)
		list(JOIN GPG_CMD " " GPG_CMD)
		string(REPLACE "$ENV" "\$" GPG_CMD "${GPG_CMD}")
		set(GPG_CMD_SH "${WORKSPACE}/.gpg")
		file(WRITE "${GPG_CMD_SH}" "#!/bin/sh\n")
		file(APPEND "${GPG_CMD_SH}" "set -euo pipefail\n")
		file(APPEND "${GPG_CMD_SH}" "${GPG_CMD} $@\n")
		file(CHMOD "${GPG_CMD_SH}" FILE_PERMISSIONS OWNER_READ OWNER_EXECUTE)
		set(GPG_CMD "${GPG_CMD_SH}")
	endif()

	if(DEFINED ENV{GITLAB_CI} AND NOT EXISTS "$ENV{HOME}/.gnupg")
		set(gnupg_private "${WORKSPACE}/gnupg-private.asc")
		download_file("${gnupg_private}" $ENV{GPG_FILE_ID})
		step(${GPG_CMD} --import ${gnupg_private})
	endif()

	set(${_var} ${GPG_CMD} PARENT_SCOPE)
endfunction()
