#
# Functions used by several mount* scripts in initscripts package
#
# Sourcer must set PATH and include /lib/init in it because
# domount() uses the custom readlink program
#
# Sourcer must also source /lib/lsb/init-functions.sh

# $1: directory
is_empty_dir() {
	for FILE in $1/* $1/.*
	do
		case "$FILE" in
		  "$1/.*") return 0 ;;
		  "$1/*"|"$1/."|"$1/..") continue ;;
		  *) return 1 ;;
		esac
	done
	return 0
}


selinux_enabled () {
	which selinuxenabled >/dev/null 2>&1 && selinuxenabled
}


# $1: file system type
# $2: alternative file system type (or empty string if none)
# $3: mount point
# $4... : extra mount program options
domount () {
	DO_MTAB=""
	MTAB_PATH="$(readlink -f /etc/mtab || :)"
	case "$MTAB_PATH" in
	  /proc/*)
		# Assume that /proc/ is not writable
		;;
	  /*)
		# Only update mtab if it is known to be writable
		# Note that the touch program is in /usr/bin
		if touch "$MTAB_PATH" >/dev/null 2>&1
		then
			DO_MTAB=y
		fi
		;;
	  "")
		[ -L /etc/mtab ] && MTAB_PATH="$(readlink /etc/mtab)"
		if [ "$MTAB_PATH" ]
		then
			log_failure_msg "Cannot initialize ${MTAB_PATH}."
		else
			log_failure_msg "Cannot initialize /etc/mtab."
		fi
		;;
	  *)
		log_failure_msg "Illegal mtab location '${MTAB_PATH}'."
		;;
	esac

	#
	# Initialize mtab file if necessary
	#
	if [ "$DO_MTAB" ] && [ ! -f /etc/mtab ]
	then
		:> /etc/mtab
		chmod 644 /etc/mtab
	fi
	if selinux_enabled && which restorecon >/dev/null 2>&1 && [ -r /etc/mtab ]
	then
		restorecon /etc/mtab
	fi

	MTPT="$3"
	KERNEL="$(uname -s)"
	# Figure out filesystem type
	FSTYPE=
	if [ "$1" = proc ]
	then
		case "$KERNEL" in
			Linux|GNU) FSTYPE=proc ;;
			*FreeBSD)  FSTYPE=linprocfs ;;
			*)         FSTYPE=procfs ;;
		esac
	elif grep -E -qs "$1\$" /proc/filesystems
	then
		FSTYPE=$1
	elif grep -E -qs "$2\$" /proc/filesystems
	then
		FSTYPE=$2
	fi

	if [ ! "$FSTYPE" ]
	then
		if [ "$2" ]
		then
			log_warning_msg "Filesystem types '$1' and '$2' are not supported. Skipping mount."
		else
			log_warning_msg "Filesystem type '$1' is not supported. Skipping mount."
		fi
		return
	fi

	# Get the options from /etc/fstab.
	OPTS=
	if [ -f /etc/fstab ]
	then
		exec 9<&0 </etc/fstab

		while read TAB_DEV TAB_MTPT TAB_FSTYPE TAB_OPTS TAB_REST
		do
			case "$TAB_DEV" in (""|\#*) continue ;; esac
			[ "$MTPT" = "$TAB_MTPT" ] || continue
			[ "$FSTYPE" = "$TAB_FSTYPE" ] || continue
			case "$TAB_OPTS" in
			  noauto|*,noauto|noauto,*|*,noauto,*)
				exec 0<&9 9<&-
				return
				;;
			  ?*)
				OPTS="-o$TAB_OPTS"
				;;
			esac
			break
		done

		exec 0<&9 9<&-
	fi

	if [ ! -d "$MTPT" ]
	then
		log_warning_msg "Mount point '$MTPT' does not exist. Skipping mount."
		return
	fi

	# We give file system type as device name
	if mountpoint -q "$MTPT"
	then
		# Already mounted
		# Just update mtab, if possible and necessary
		if [ "$DO_MTAB" ] && ! grep -E -sq "^([^ ]+) +$MTPT +" /etc/mtab
		then
			mount -f -t $FSTYPE $OPTS $4 $FSTYPE $MTPT
		fi
	else
		# Not already mounted
		is_empty_dir "$MTPT" >/dev/null 2>&1 || log_warning_msg "Files under mount point '$MTPT' will be hidden."
		# Mount it, updating mtab if possible
		if [ "$DO_MTAB" ]
		then
			mount -t $FSTYPE $OPTS $4 $FSTYPE $MTPT
		else
			mount -n -t $FSTYPE $OPTS $4 $FSTYPE $MTPT
		fi
	fi
}

