/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.commons.converter;

import javax.faces.application.FacesMessage;
import javax.faces.component.PartialStateHolder;
import javax.faces.component.UIComponent;
import javax.faces.context.FacesContext;
import javax.faces.convert.Converter;
import javax.faces.convert.ConverterException;

import org.apache.myfaces.buildtools.maven2.plugin.builder.annotation.JSFConverter;
import org.apache.myfaces.buildtools.maven2.plugin.builder.annotation.JSFProperty;
import org.apache.myfaces.commons.util.MessageUtils;

/**
 * Converts a Java 5 Enum.
 * 
 * see Javadoc of <a href="http://java.sun.com/j2ee/javaserverfaces/1.2/docs/api/index.html">JSF Specification</a>
 *
 *   
 * @author Stan Silvert
 */
@JSFConverter(
   name = "mcc:convertEnum",
   tagClass = "org.apache.myfaces.commons.converter.ConvertEnumTag",
   tagHandler = "org.apache.myfaces.commons.converter.ConvertEnumTagHandler",
   serialuidtag = "3864584277821896141L")
public class EnumConverter implements Converter, PartialStateHolder
{
    
    public static final String CONVERTER_ID = "org.apache.myfaces.commons.converter.Enum";

    public static final String ENUM_ID
            = "org.apache.myfaces.commons.converter.EnumConverter.ENUM";

    public static final String ENUM_NO_CLASS_ID
            = "org.apache.myfaces.commons.converter.EnumConverter.ENUM_NO_CLASS";

    public static final String ENUM_NO_ENUM_CLASS
            = "org.apache.myfaces.commons.converter.EnumConverter.ENUM_NO_ENUM_CLASS";

    public static final String SEPARATOR = "#";

    private Class targetClass;
    
    private boolean isTransient = false;
    
    /** Creates a new instance of EnumConverter */
    public EnumConverter()
    {
    }
    
    public EnumConverter(Class targetClass)
    {
        if (!targetClass.isEnum())
        {
            throw new IllegalArgumentException("targetClass for EnumConverter must be an Enum");
        }
        this.targetClass = targetClass;
    }

    public String getAsString(FacesContext facesContext, UIComponent uiComponent, Object value)
            throws ConverterException
    {
        if (facesContext == null)
        {
            throw new NullPointerException("facesContext can not be null");
        }
        if (uiComponent == null)
        {
            throw new NullPointerException("uiComponent can not be null");
        }
        if (value == null)
        {
            return "";
        }

      Class converterClass = null;
      String idPrefix = "";

      if (targetClass != null)
      {
        converterClass = targetClass;
      }
      else if (value.getClass().isEnum())
      {
        converterClass = value.getClass();
        idPrefix = value.getClass().getName() + SEPARATOR;
      }
      else if (value.getClass().getDeclaringClass() != null && value.getClass().getDeclaringClass().isEnum())
      {
        converterClass = value.getClass().getDeclaringClass();
        idPrefix = value.getClass().getDeclaringClass().getName() + SEPARATOR;
      }

      if (converterClass == null || targetClass != null)
      {
        checkTargetClass(facesContext, uiComponent, value, converterClass);
      }

      for (Object enumConstant : converterClass.getEnumConstants())
      {
          if (enumConstant == value)
          {
              return idPrefix + ((Enum)enumConstant).name();
          }
      }

        return value.toString();
    }

    public Object getAsObject(FacesContext facesContext, UIComponent uiComponent, String value)
            throws ConverterException
    {
        if (facesContext == null)
        {
            throw new NullPointerException("facesContext");
        }
        if (uiComponent == null)
        {
            throw new NullPointerException("uiComponent");
        }
        if (value == null)
        {
            return null;
        }
        value = value.trim();
        if (value.length() == 0)
        {
            return null;
        }

      Class converterClass = null;
      String enumName = null;

      if (targetClass != null)
      {
          enumName = value;
          converterClass = targetClass;
      }
      else if (value.indexOf(SEPARATOR) > 0)
      {
          int index = value.indexOf(SEPARATOR);
          if (index < value.length() - 2)
          {
              String className = value.substring(0, index);
              enumName = value.substring(index +1);
              try
              {
                  converterClass = Class.forName(className);
              }
              catch (ClassNotFoundException e)
              {
                  //          LOG.error("Catched: " + e.getMessage(), e);
              }
          }
          if (enumName == null || enumName.length() == 0)
          {
              return null;
          }
      }

      if (converterClass == null)
      {
          checkTargetClass(facesContext, uiComponent, value, converterClass);
      }

        // we know targetClass and value can't be null, so we can use Enum.valueOf
        // instead of the hokey looping called for in the javadoc
        try
        {
          //noinspection unchecked
          return Enum.valueOf(converterClass, enumName);
        }
        catch (IllegalArgumentException e)
        {
            Object[] params = new Object[]{value, 
                                           firstConstantOfEnum(), 
                                           MessageUtils.getLabel(facesContext, uiComponent)};
            
            throw new ConverterException(MessageUtils.getMessage(FacesMessage.SEVERITY_ERROR,
                                                                       ENUM_ID,
                                                                       params));
        }
    }

    private void checkTargetClass(FacesContext facesContext, UIComponent uiComponent,
                                  Object value, Class converterClass)
    {
        if (converterClass == null)
        {
            Object[] params = new Object[]{value, MessageUtils.getLabel(facesContext, uiComponent)};
            throw new ConverterException(MessageUtils.getMessage(FacesMessage.SEVERITY_ERROR,
                                                                       ENUM_NO_CLASS_ID, 
                                                                       params));
        }
        if (!converterClass.isEnum())
        {
            Object[] params = new Object[]{value, MessageUtils.getLabel(facesContext, uiComponent)};
            throw new ConverterException(MessageUtils.getMessage(FacesMessage.SEVERITY_ERROR,
                                                                       ENUM_NO_ENUM_CLASS, 
                                                                       params));
        }
    }

    // find the first constant value of the targetClass and return as a String
    private String firstConstantOfEnum()
    {
        Object[] enumConstants= targetClass.getEnumConstants();

        if (enumConstants.length != 0 )
        {
            return enumConstants[0].toString();
        }
        
        return ""; // if empty Enum
    }

    public void restoreState(FacesContext context, Object state)
    {
        if (state != null)
        {
            targetClass = (Class)state;
        }
    }

    public Object saveState(FacesContext context)
    {
        if (!initialStateMarked())
        {
            return targetClass;
        }
        return null;
    }

    public void setTransient(boolean newTransientValue)
    {
        isTransient = newTransientValue;
    }

    public boolean isTransient()
    {
        return isTransient;
    }

    /**
     * The enum class to be used for this converter as reference
     * 
     */
    @JSFProperty
    public Class getTargetClass()
    {
        return targetClass;
    }

    public void setTargetClass(Class targetClass)
    {
        this.targetClass = targetClass;
        clearInitialState();
    }
    
    private boolean _initialStateMarked = false;

    public void clearInitialState()
    {
        _initialStateMarked = false;
    }

    public boolean initialStateMarked()
    {
        return _initialStateMarked;
    }

    public void markInitialState()
    {
        _initialStateMarked = true;
    }
}
