<!--
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements.  See the NOTICE file
distributed with this work for additional information
regarding copyright ownership.  The ASF licenses this file
to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.
-->
# Maven Artifact Resolver Ant Tasks

The Maven Artifact Resolver Ant Tasks enable build scripts for [Apache Ant](http://ant.apache.org/) 1.7+ to use
[Maven Artifact Resolver](https://maven.apache.org/resolver/) combined to
[Apache Maven Artifact Resolver Provider](https://maven.apache.org/ref/current/maven-resolver-provider/)
to resolve dependencies and install and deploy locally built artifacts.

To integrate the tasks into your build file, copy the Über JAR into your project's lib directory and use the following
snippet to load it:

```xml
<project xmlns:resolver="antlib:org.apache.maven.resolver.ant" ...>
  <taskdef uri="antlib:org.apache.maven.resolver.ant" resource="org/apache/maven/resolver/ant/antlib.xml"
    classpath="lib/maven-resolver-ant-tasks-${project.version}-uber.jar" />
  ...
</project>
```

See the `build.xml` in the project sources for a complete example build script.

## Settings

The Ant tasks are tightly integrated with the usual [Apache Maven settings.xml](/settings.html).
By default, the usual `${user.home}/.m2/settings.xml` is used for user settings.

For the global settings, different paths will be tried:

* `${ant.home}/etc/settings.xml`
* `${maven.home}/conf/settings.xml`

The `<settings/>` definition is used to change that:

```xml
<settings file="my-settings.xml" globalfile="myglobal-settings.xml"/>
```

Some settings defined in the settings file or in the POM can also be changed inside the Ant file.

### Proxy Settings

Proxy definitions are used throughout the whole session. There may be multiple
proxies set. The proxy to use will be chosen by evaluating the `nonProxyHosts` on
each proxy definition, the first matching proxy will be used for a given remote connection.

```xml
<proxy host="proxy.mycorp.com" port="8080" type="http" nonProxyHosts="127.*|localhost|*.mycorp.com"/>
```

### Authentication

Authentication elements are used to access remote repositories. Every
authentication definition will be added globally and chosen based on the
`servers` attribute. If this attribute is not set, an authentication has to be
referenced explicitly to be used.

```xml
<authentication username="login" password="pw" id="auth"/>
<authentication privateKeyFile="file.pk" passphrase="phrase" servers="distrepo" id="distauth"/>
```

### Local Repository

Only one local repository can be used at a time.

```xml
<localrepo dir="someDir"/>
```

### Remote Repositories

Remote repositories may be defined directly:

```xml
<remoterepo id="ossrh" url="https://oss.sonatype.org/content/repositories/snapshots/" type="default" releases="false" snapshots="true" updates="always" checksums="fail"/>

<remoterepo id="rao" url="https://repository.apache.org/content/groups/public/">
    <releases enabled="true" updates="daily" checksums="warn"/>
    <snapshots enabled="false"/>
    <authentication refid="auth"/>
</remoterepo>

<remoterepo id="distrepo" url="..." authref="distauth"/>
```

Multiple repositories may be used as a group in every place that is legal for a
remote repository:

```xml
<remoterepos id="all">
    <remoterepo refid="ossrh"/>
    <remoterepo refid="rao"/>
    <remoterepo refid="distrepo"/>
</remoterepos>
```

*Note:* Currently, only file:, http: and https: protocols are supported for remote repositories.

### Mirrors

```xml
<mirror id="" url="" mirrorOf=""/>
```

### Offline Mode

To suppress any network activity and only use already cached artifacts/metadata, you can use a boolean property:

```xml
<property name="resolver.offline" value="true"/>
```

## Project

Project settings deal with locally available information about the build.

### POM

The POM is the data type used to determine the target for the install and
deploy tasks. If you define a POM without an id based on a full `pom.xml` file,
that POM will be used by default for install and deploy.

```xml
<pom file="pom.xml" id="pom"/>
<pom groupId="g" artifactId="a" version="v"/>
<pom coords="g:a:v"/>
```

#### Properties

If a POM is set via a file parameter its effective model is made available as properties to the Ant project.
The properties are prefixed with the ref id of the `<pom>` element, e.g. `${pom.version}` for the example above.
Likewise, project properties defined in the POM are accessible via the prefix `pom.properties.`. If no id has been
assigned, the properties use the prefix `pom.` by default.

### Output Artifacts

`<artifact>` elements define the artifacts produced by this build that should be installed or deployed.

```xml
<artifact file="file-src.jar" type="jar" classifier="sources" id="src"/>

<artifacts id="producedArtifacts">
    <artifact refid="src"/>
    <artifact file="file-src.jar"/>
</artifacts>
```

### Dependencies

Dependencies are used to to create classpaths or filesets. They are used by
the `<resolve>`-task, which collects the artifacts belonging to the dependencies
transitively.

```xml
<dependency coords="g:a:v:scope"/>

<dependency groupId="g" artifactId="a" version="v" classifier="c" type="jar" scope="runtime">
    <exclusion coords="g:a"/>
    <exclusion groupId="g" artifactId="a"/>
</dependency>

<dependencies id="deps">
    <dependency refid="first"/>
    <dependency refid="second"/>
    <exclusion coords="g:a"/> <!-- global exclusion for all dependencies of this group -->
</dependencies>

<dependencies>
    <dependency coords="test:artifact:1.0:runtime"/>
    <dependencies refid="deps"/> <!-- nested dependency collection merged into this one -->
</dependencies>

<dependencies id="depsFromPom" pomRef="pom"/>

<dependencies id="depsFromPlainTextFile" file="dependencies.txt"/>
<!--
Each non-empty line of that text file declares one dependency, using the same syntax as for the `coords` attribute
of the `<dependency>` element, i.e.
<groupId>:<artifactId>:<version>[[:<type>[:<classifier>]]:<scope>]
Everything after the first hash (#) character on a line is considered a comment.
-->
```

## Tasks

### Install

You need to set a POM that references a file for the install task to work.

```xml
<install artifactsref="producedArtifacts"/>
```

### Deploy

You need to set a POM that references a file for the deploy task to work, as that POM file will be deployed to repository.

```xml
<deploy artifactsref="producedArtifacts">
    <remoterepo refid="distrepo"/>
    <snapshotrepo refid="snaprepo">
</deploy>
```

### Resolve

The `<resolve>`-task is used to collect and resolve dependencies from remote
servers. If no repositories are set explicitly for the task, the repositories
referenced by "resolver.repositories" are used. This contains only central by
default, but can be overridden by supplying another repository definition with
this id.


This task is able to assemble the collected dependencies in three different ways:

* Classpath: The `<path>` element defines a classpath with all resolved dependencies.
* Files: `<files>` will assemble a resource collection containing all resolved dependencies and/or copy the files to some directory.
* Properties: `<properties>` will set properties with the given prefix and the coordinates to the path to the resolved file.

These targets may also be mentioned more than once for the same resolve task,
but only one `<dependencies>` element is allowed.

```xml
<resolve failOnMissingAttachments="true">
    <dependencies>
        <dependency coords="org.apache.maven:maven-profile:2.0.6"/>
        <exclusion artifactId="junit"/>
        <exclusion groupId="org.codehaus.plexus"/>
    </dependencies>
    <path refid="cp" classpath="compile"/>
    <files refid="src.files" attachments="sources" dir="target/sources"
           layout="{artifactId}-{classifier}.{extension}"/>
    <files refid="api.files" attachments="javadoc" dir="target/javadoc"
           layout="{artifactId}-{classifier}.{extension}"/>
    <properties prefix="dep." scopes="provided,system"/>
</resolve>

<resolve dependenciesref="deps">
    <path refid="cp.compile" classpath="compile"/>
    <path refid="cp.test" classpath="test"/>
</resolve>
```

Scope filters can be set on every target, enumerating included and/or excluded
scope names. Exclusions are denoted by prefixing the scope name with `-` or `!` (e.g. `provided,!system`).

The `classpath` attribute is a shortcut for the scope filters (e.g.
`classpath="compile"` equals `scope="provided,system,compile"`). Valid values are
"`compile`", "`runtime`", "`test`".

```xml
<resolve>
    <dependencies pomRef="pom"/>
    <remoterepos refid="all"/>
    <path refid="cp" classpath="compile"/>
    <path refid="tp" classpath="test"/>
</resolve>
```

The layout attribute of the `<files>` element is only allowed when the `dir` attribute is also given and recognizes the
following placeholders to refer to the coordinates of the currently processed artifact:

* `{groupId}`, e.g. "org.apache.maven.resolver"
* `{groupIdDirs}`, e.g. "org/apache/maven/resolver"
* `{artifactId}`, e.g. "maven-resolver-api"
* `{version}`, e.g. "1.0.0-20140518.181353-123"
* `{baseVersion}`, e.g. "1.0.0-SNAPSHOT"
* `{extension}`, e.g. "jar"
* `{classifier}`, e.g. "sources"

# More information
See [usage.md](https://github.com/apache/maven-resolver-ant-tasks/blob/master/usage.md) for info.

The [examples](https://github.com/apache/maven-resolver-ant-tasks/tree/master/examples) contains 7 complete examples of various ways to use Maven Resolver Ant Tasks.