/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.backend.store.cassandra;

import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.KeyspaceMetadata;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.exceptions.DriverException;
import com.datastax.driver.core.exceptions.InvalidQueryException;
import com.datastax.driver.core.schemabuilder.DropKeyspace;
import com.datastax.driver.core.schemabuilder.KeyspaceOptions;
import com.datastax.driver.core.schemabuilder.SchemaBuilder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hugegraph.HugeException;
import org.apache.hugegraph.backend.BackendException;
import org.apache.hugegraph.backend.id.Id;
import org.apache.hugegraph.backend.query.Query;
import org.apache.hugegraph.backend.serializer.MergeIterator;
import org.apache.hugegraph.backend.serializer.TableBackendEntry;
import org.apache.hugegraph.backend.store.AbstractBackendStore;
import org.apache.hugegraph.backend.store.BackendAction;
import org.apache.hugegraph.backend.store.BackendEntry;
import org.apache.hugegraph.backend.store.BackendFeatures;
import org.apache.hugegraph.backend.store.BackendMutation;
import org.apache.hugegraph.backend.store.BackendSession;
import org.apache.hugegraph.backend.store.BackendStore;
import org.apache.hugegraph.backend.store.BackendStoreProvider;
import org.apache.hugegraph.backend.store.cassandra.CassandraBackendEntry;
import org.apache.hugegraph.backend.store.cassandra.CassandraFeatures;
import org.apache.hugegraph.backend.store.cassandra.CassandraMetrics;
import org.apache.hugegraph.backend.store.cassandra.CassandraOptions;
import org.apache.hugegraph.backend.store.cassandra.CassandraSessionPool;
import org.apache.hugegraph.backend.store.cassandra.CassandraTable;
import org.apache.hugegraph.backend.store.cassandra.CassandraTables;
import org.apache.hugegraph.config.CoreOptions;
import org.apache.hugegraph.config.HugeConfig;
import org.apache.hugegraph.config.TypedOption;
import org.apache.hugegraph.exception.ConnectionException;
import org.apache.hugegraph.type.HugeType;
import org.apache.hugegraph.util.E;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public abstract class CassandraStore
extends AbstractBackendStore<CassandraSessionPool.Session> {
    private static final Logger LOG = Log.logger(CassandraStore.class);
    private static final BackendFeatures FEATURES = new CassandraFeatures();
    private final String store;
    private final String keyspace;
    private final BackendStoreProvider provider;
    private final Map<String, CassandraTable> tables;
    private CassandraSessionPool sessions;
    private HugeConfig conf;
    private boolean isGraphStore;

    public CassandraStore(BackendStoreProvider provider, String keyspace, String store) {
        E.checkNotNull((Object)keyspace, (String)"keyspace");
        E.checkNotNull((Object)store, (String)"store");
        this.provider = provider;
        this.keyspace = keyspace;
        this.store = store;
        this.tables = new ConcurrentHashMap<String, CassandraTable>();
        this.sessions = null;
        this.conf = null;
        this.registerMetaHandlers();
        LOG.debug("Store loaded: {}", (Object)store);
    }

    private void registerMetaHandlers() {
        this.registerMetaHandler("metrics", (session, meta, args) -> {
            CassandraMetrics metrics = this.createMetrics(this.conf, this.sessions, this.keyspace);
            return metrics.metrics();
        });
        this.registerMetaHandler("compact", (session, meta, args) -> {
            CassandraMetrics metrics = this.createMetrics(this.conf, this.sessions, this.keyspace);
            return metrics.compact();
        });
    }

    protected CassandraMetrics createMetrics(HugeConfig conf, CassandraSessionPool sessions, String keyspace) {
        return new CassandraMetrics(conf, sessions, keyspace);
    }

    protected void registerTableManager(HugeType type, CassandraTable table) {
        this.registerTableManager(type.string(), table);
    }

    protected void registerTableManager(String name, CassandraTable table) {
        this.tables.put(name, table);
    }

    protected void unregisterTableManager(String name) {
        this.tables.remove(name);
    }

    public String store() {
        return this.store;
    }

    public String database() {
        return this.keyspace;
    }

    public BackendStoreProvider provider() {
        return this.provider;
    }

    public synchronized void open(HugeConfig config) {
        LOG.debug("Store open: {}", (Object)this.store);
        E.checkNotNull((Object)config, (String)"config");
        if (this.sessions == null) {
            this.sessions = new CassandraSessionPool(config, this.keyspace, this.store);
        }
        assert (this.sessions != null);
        if (!this.sessions.closed()) {
            LOG.debug("Store {} has been opened before", (Object)this.store);
            this.sessions.useSession();
            return;
        }
        this.conf = config;
        String graphStore = (String)this.conf.get((TypedOption)CoreOptions.STORE_GRAPH);
        this.isGraphStore = this.store.equals(graphStore);
        this.sessions.open();
        try {
            LOG.debug("Store connect with keyspace: {}", (Object)this.keyspace);
            try {
                this.sessions.session().open();
            }
            catch (InvalidQueryException e) {
                if (!e.getMessage().contains(String.format("Keyspace '%s' does not exist", this.keyspace))) {
                    throw e;
                }
                if (this.isSchemaStore()) {
                    LOG.info("Failed to connect keyspace: {}, try to init keyspace later", (Object)this.keyspace);
                }
            }
        }
        catch (Throwable e) {
            try {
                this.sessions.close();
            }
            catch (Throwable e2) {
                LOG.warn("Failed to close cluster after an error", e2);
            }
            throw new ConnectionException("Failed to connect to Cassandra", e);
        }
        LOG.debug("Store opened: {}", (Object)this.store);
    }

    public void close() {
        LOG.debug("Store close: {}", (Object)this.store);
        this.sessions.close();
    }

    public boolean opened() {
        this.checkClusterConnected();
        return this.sessions.session().opened();
    }

    public void mutate(BackendMutation mutation) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Store {} mutation: {}", (Object)this.store, (Object)mutation);
        }
        this.checkOpened();
        CassandraSessionPool.Session session = this.sessions.session();
        Iterator it = mutation.mutation();
        while (it.hasNext()) {
            this.mutate(session, (BackendAction)it.next());
        }
    }

    private void mutate(CassandraSessionPool.Session session, BackendAction item) {
        CassandraBackendEntry entry = CassandraStore.castBackendEntry(item.entry());
        if (!entry.selfChanged() && entry.subRows().isEmpty()) {
            LOG.warn("The entry will be ignored due to no change: {}", (Object)entry);
        }
        CassandraTable table = !entry.olap() ? this.table(entry.type()) : (entry.type().isIndex() ? this.table(this.olapTableName(entry.type())) : this.table(this.olapTableName(entry.subId())));
        switch (item.action()) {
            case INSERT: {
                if (entry.selfChanged()) {
                    table.insert(session, entry.row());
                }
                for (TableBackendEntry.Row row : entry.subRows()) {
                    this.table(row.type()).insert(session, row);
                }
                break;
            }
            case DELETE: {
                if (entry.selfChanged()) {
                    table.delete(session, entry.row());
                }
                for (TableBackendEntry.Row row : entry.subRows()) {
                    this.table(row.type()).delete(session, row);
                }
                break;
            }
            case APPEND: {
                if (entry.selfChanged()) {
                    table.append(session, entry.row());
                }
                for (TableBackendEntry.Row row : entry.subRows()) {
                    this.table(row.type()).append(session, row);
                }
                break;
            }
            case ELIMINATE: {
                if (entry.selfChanged()) {
                    table.eliminate(session, entry.row());
                }
                for (TableBackendEntry.Row row : entry.subRows()) {
                    this.table(row.type()).eliminate(session, row);
                }
                break;
            }
            case UPDATE_IF_PRESENT: {
                if (entry.selfChanged()) {
                    table.updateIfPresent((BackendSession)session, entry.row());
                }
                assert (entry.subRows().isEmpty()) : entry.subRows();
                break;
            }
            case UPDATE_IF_ABSENT: {
                if (entry.selfChanged()) {
                    table.updateIfAbsent((BackendSession)session, entry.row());
                }
                assert (entry.subRows().isEmpty()) : entry.subRows();
                break;
            }
            default: {
                throw new AssertionError((Object)String.format("Unsupported mutate action: %s", item.action()));
            }
        }
    }

    public Iterator<BackendEntry> query(Query query) {
        this.checkOpened();
        HugeType type = CassandraTable.tableType((Query)query);
        String tableName = query.olap() ? this.olapTableName(type) : type.string();
        CassandraTable table = this.table(tableName);
        MergeIterator entries = table.query(this.session(null), query);
        Set olapPks = query.olapPks();
        if (this.isGraphStore && !olapPks.isEmpty()) {
            ArrayList<Iterator<BackendEntry>> iterators = new ArrayList<Iterator<BackendEntry>>();
            for (Id pk : olapPks) {
                Query q = query.copy();
                table = this.table(this.olapTableName(pk));
                iterators.add(table.query(this.session(null), q));
            }
            entries = new MergeIterator(entries, iterators, BackendEntry::mergeable);
        }
        return entries;
    }

    public Number queryNumber(Query query) {
        this.checkOpened();
        CassandraTable table = this.table(CassandraTable.tableType((Query)query));
        return table.queryNumber(this.sessions.session(), query);
    }

    public BackendFeatures features() {
        return FEATURES;
    }

    public void init() {
        this.checkClusterConnected();
        if (!this.existsKeyspace()) {
            this.initKeyspace();
        }
        if (this.sessions.session().opened()) {
            LOG.warn("Session has ever been opened(exist keyspace '{}' before)", (Object)this.keyspace);
        } else {
            this.sessions.session().open();
        }
        this.checkOpened();
        this.initTables();
        LOG.debug("Store initialized: {}", (Object)this.store);
    }

    public void clear(boolean clearSpace) {
        this.checkClusterConnected();
        if (this.existsKeyspace()) {
            if (!clearSpace) {
                this.checkOpened();
                this.clearTables();
            } else {
                this.clearKeyspace();
            }
        }
        LOG.debug("Store cleared: {}", (Object)this.store);
    }

    public boolean initialized() {
        this.checkClusterConnected();
        if (!this.existsKeyspace()) {
            return false;
        }
        for (CassandraTable table : this.tables()) {
            if (this.existsTable(table.table())) continue;
            return false;
        }
        return true;
    }

    public void truncate() {
        this.checkOpened();
        this.truncateTables();
        LOG.debug("Store truncated: {}", (Object)this.store);
    }

    public void beginTx() {
        this.checkOpened();
        CassandraSessionPool.Session session = this.sessions.session();
        if (session.txState() != BackendStore.TxState.CLEAN) {
            LOG.warn("Store {} expect state CLEAN than {} when begin()", (Object)this.store, (Object)session.txState());
        }
        session.txState(BackendStore.TxState.BEGIN);
    }

    public void commitTx() {
        this.checkOpened();
        CassandraSessionPool.Session session = this.sessions.session();
        if (session.txState() != BackendStore.TxState.BEGIN) {
            LOG.warn("Store {} expect state BEGIN than {} when commit()", (Object)this.store, (Object)session.txState());
        }
        if (!session.hasChanges()) {
            session.txState(BackendStore.TxState.CLEAN);
            LOG.debug("Store {} has nothing to commit", (Object)this.store);
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Store {} commit {} statements: {}", new Object[]{this.store, session.statements().size(), session.statements()});
        }
        session.txState(BackendStore.TxState.COMMITTING);
        try {
            session.commit();
            session.txState(BackendStore.TxState.CLEAN);
        }
        catch (DriverException e) {
            session.txState(BackendStore.TxState.COMMITT_FAIL);
            LOG.error("Failed to commit statements due to:", (Throwable)e);
            assert (session.statements().size() > 0);
            throw new BackendException("Failed to commit %s statements: '%s'...", (Throwable)e, new Object[]{session.statements().size(), session.statements().iterator().next()});
        }
    }

    public void rollbackTx() {
        this.checkOpened();
        CassandraSessionPool.Session session = this.sessions.session();
        if (session.txState() != BackendStore.TxState.COMMITT_FAIL && session.txState() != BackendStore.TxState.CLEAN) {
            LOG.warn("Store {} expect state COMMIT_FAIL/COMMITTING/CLEAN than {} when rollback()", (Object)this.store, (Object)session.txState());
        }
        session.txState(BackendStore.TxState.ROLLBACKING);
        try {
            session.rollback();
        }
        finally {
            session.txState(BackendStore.TxState.CLEAN);
        }
    }

    protected Cluster cluster() {
        return this.sessions.cluster();
    }

    protected boolean existsKeyspace() {
        return this.cluster().getMetadata().getKeyspace(this.keyspace) != null;
    }

    protected boolean existsTable(String table) {
        KeyspaceMetadata keyspace = this.cluster().getMetadata().getKeyspace(this.keyspace);
        return keyspace != null && keyspace.getTable(table) != null;
    }

    protected void initKeyspace() {
        KeyspaceOptions stmt = SchemaBuilder.createKeyspace((String)this.keyspace).ifNotExists().with().replication(CassandraStore.parseReplica(this.conf));
        LOG.debug("Create keyspace: {}", (Object)stmt);
        Session session = this.cluster().connect();
        try {
            session.execute((Statement)stmt);
        }
        finally {
            if (!session.isClosed()) {
                session.close();
            }
        }
    }

    private static Map<String, Object> parseReplica(HugeConfig conf) {
        HashMap<String, Object> replication = new HashMap<String, Object>();
        String strategy = (String)conf.get(CassandraOptions.CASSANDRA_STRATEGY);
        replication.put("class", strategy);
        switch (strategy) {
            case "SimpleStrategy": {
                List replicas = (List)conf.get(CassandraOptions.CASSANDRA_REPLICATION);
                E.checkArgument((replicas.size() == 1 ? 1 : 0) != 0, (String)"Individual factor value should be provided with SimpleStrategy for Cassandra", (Object[])new Object[0]);
                int factor = CassandraStore.convertFactor((String)replicas.get(0));
                replication.put("replication_factor", factor);
                break;
            }
            case "NetworkTopologyStrategy": {
                Map replicaMap = conf.getMap(CassandraOptions.CASSANDRA_REPLICATION);
                for (Map.Entry e : replicaMap.entrySet()) {
                    E.checkArgument((!((String)e.getKey()).isEmpty() ? 1 : 0) != 0, (String)"The datacenter can't be empty", (Object[])new Object[0]);
                    replication.put((String)e.getKey(), CassandraStore.convertFactor((String)e.getValue()));
                }
                break;
            }
            default: {
                throw new AssertionError((Object)String.format("Illegal replication strategy '%s', valid strategy is 'SimpleStrategy' or 'NetworkTopologyStrategy'", strategy));
            }
        }
        return replication;
    }

    private static int convertFactor(String factor) {
        try {
            return Integer.valueOf(factor);
        }
        catch (NumberFormatException e) {
            throw new BackendException("Expect int factor value for SimpleStrategy, but got '%s'", new Object[]{factor});
        }
    }

    protected void clearKeyspace() {
        DropKeyspace stmt = SchemaBuilder.dropKeyspace((String)this.keyspace).ifExists();
        LOG.debug("Drop keyspace: {}", (Object)stmt);
        Session session = this.cluster().connect();
        try {
            session.execute((Statement)stmt);
        }
        finally {
            if (!session.isClosed()) {
                session.close();
            }
        }
    }

    protected void initTables() {
        CassandraSessionPool.Session session = this.sessions.session();
        for (CassandraTable table : this.tables()) {
            table.init((BackendSession)session);
        }
    }

    protected void clearTables() {
        CassandraSessionPool.Session session = this.sessions.session();
        for (CassandraTable table : this.tables()) {
            table.clear(session);
        }
    }

    protected void truncateTables() {
        CassandraSessionPool.Session session = this.sessions.session();
        for (CassandraTable table : this.tables()) {
            if (table.isOlap()) {
                table.dropTable(session);
                continue;
            }
            table.truncate(session);
        }
    }

    protected Collection<CassandraTable> tables() {
        return this.tables.values();
    }

    protected final CassandraTable table(HugeType type) {
        return this.table(type.string());
    }

    protected final CassandraTable table(String name) {
        assert (name != null);
        CassandraTable table = this.tables.get(name);
        if (table == null) {
            throw new BackendException("Unsupported table: %s", new Object[]{name});
        }
        return table;
    }

    protected CassandraSessionPool.Session session(HugeType type) {
        this.checkOpened();
        return this.sessions.session();
    }

    protected final void checkClusterConnected() {
        E.checkState((this.sessions != null && this.sessions.clusterConnected() ? 1 : 0) != 0, (String)"Cassandra cluster has not been connected", (Object[])new Object[0]);
    }

    protected static final CassandraBackendEntry castBackendEntry(BackendEntry entry) {
        assert (entry instanceof CassandraBackendEntry) : entry.getClass();
        if (!(entry instanceof CassandraBackendEntry)) {
            throw new BackendException("Cassandra store only supports CassandraBackendEntry");
        }
        return (CassandraBackendEntry)entry;
    }

    public static class CassandraSystemStore
    extends CassandraGraphStore {
        private final CassandraTables.Meta meta = new CassandraTables.Meta();

        public CassandraSystemStore(BackendStoreProvider provider, String keyspace, String store) {
            super(provider, keyspace, store);
        }

        @Override
        public void init() {
            super.init();
            this.checkOpened();
            String driverVersion = this.provider().driverVersion();
            this.meta.writeVersion(this.session(null), driverVersion);
            LOG.info("Write down the backend version: {}", (Object)driverVersion);
        }

        public String storedVersion() {
            CassandraSessionPool.Session session = this.session(null);
            return this.meta.readVersion(session);
        }

        @Override
        protected Collection<CassandraTable> tables() {
            ArrayList<CassandraTable> tables = new ArrayList<CassandraTable>(super.tables());
            tables.add(this.meta);
            return tables;
        }
    }

    public static class CassandraGraphStore
    extends CassandraStore {
        public CassandraGraphStore(BackendStoreProvider provider, String keyspace, String store) {
            super(provider, keyspace, store);
            this.registerTableManager(HugeType.VERTEX, (CassandraTable)new CassandraTables.Vertex(store));
            this.registerTableManager(HugeType.EDGE_OUT, CassandraTables.Edge.out(store));
            this.registerTableManager(HugeType.EDGE_IN, CassandraTables.Edge.in(store));
            this.registerTableManager(HugeType.SECONDARY_INDEX, (CassandraTable)new CassandraTables.SecondaryIndex(store));
            this.registerTableManager(HugeType.RANGE_INT_INDEX, (CassandraTable)new CassandraTables.RangeIntIndex(store));
            this.registerTableManager(HugeType.RANGE_FLOAT_INDEX, (CassandraTable)new CassandraTables.RangeFloatIndex(store));
            this.registerTableManager(HugeType.RANGE_LONG_INDEX, (CassandraTable)new CassandraTables.RangeLongIndex(store));
            this.registerTableManager(HugeType.RANGE_DOUBLE_INDEX, (CassandraTable)new CassandraTables.RangeDoubleIndex(store));
            this.registerTableManager(HugeType.SEARCH_INDEX, (CassandraTable)new CassandraTables.SearchIndex(store));
            this.registerTableManager(HugeType.SHARD_INDEX, (CassandraTable)new CassandraTables.ShardIndex(store));
            this.registerTableManager(HugeType.UNIQUE_INDEX, (CassandraTable)new CassandraTables.UniqueIndex(store));
            this.registerTableManager(this.olapTableName(HugeType.SECONDARY_INDEX), (CassandraTable)new CassandraTables.OlapSecondaryIndex(store));
            this.registerTableManager(this.olapTableName(HugeType.RANGE_INT_INDEX), (CassandraTable)new CassandraTables.OlapRangeIntIndex(store));
            this.registerTableManager(this.olapTableName(HugeType.RANGE_LONG_INDEX), (CassandraTable)new CassandraTables.OlapRangeLongIndex(store));
            this.registerTableManager(this.olapTableName(HugeType.RANGE_FLOAT_INDEX), (CassandraTable)new CassandraTables.OlapRangeFloatIndex(store));
            this.registerTableManager(this.olapTableName(HugeType.RANGE_DOUBLE_INDEX), (CassandraTable)new CassandraTables.OlapRangeDoubleIndex(store));
        }

        public Id nextId(HugeType type) {
            throw new UnsupportedOperationException("CassandraGraphStore.nextId()");
        }

        public void increaseCounter(HugeType type, long num) {
            throw new UnsupportedOperationException("CassandraGraphStore.increaseCounter()");
        }

        public long getCounter(HugeType type) {
            throw new UnsupportedOperationException("CassandraGraphStore.getCounter()");
        }

        public boolean isSchemaStore() {
            return false;
        }

        public void checkAndRegisterOlapTable(Id id) {
            CassandraTables.Olap table = new CassandraTables.Olap(this.store(), id);
            if (!this.existsTable(table.table())) {
                throw new HugeException("Not exist table '%s'", new Object[]{table.table()});
            }
            this.registerTableManager(this.olapTableName(id), (CassandraTable)table);
        }

        public void createOlapTable(Id id) {
            CassandraTables.Olap table = new CassandraTables.Olap(this.store(), id);
            table.init((BackendSession)this.session(null));
            this.registerTableManager(this.olapTableName(id), (CassandraTable)table);
        }

        public void clearOlapTable(Id id) {
            String name = this.olapTableName(id);
            CassandraTable table = this.table(name);
            if (!this.existsTable(table.table())) {
                throw new HugeException("Not exist table '%s'", new Object[]{name});
            }
            table.truncate(this.session(null));
        }

        public void removeOlapTable(Id id) {
            String name = this.olapTableName(id);
            CassandraTable table = this.table(name);
            if (!this.existsTable(table.table())) {
                throw new HugeException("Not exist table '%s'", new Object[]{name});
            }
            table.dropTable(this.session(null));
            this.unregisterTableManager(name);
        }
    }

    public static class CassandraSchemaStore
    extends CassandraStore {
        private final CassandraTables.Counters counters = new CassandraTables.Counters();

        public CassandraSchemaStore(BackendStoreProvider provider, String keyspace, String store) {
            super(provider, keyspace, store);
            this.registerTableManager(HugeType.VERTEX_LABEL, (CassandraTable)new CassandraTables.VertexLabel());
            this.registerTableManager(HugeType.EDGE_LABEL, (CassandraTable)new CassandraTables.EdgeLabel());
            this.registerTableManager(HugeType.PROPERTY_KEY, (CassandraTable)new CassandraTables.PropertyKey());
            this.registerTableManager(HugeType.INDEX_LABEL, (CassandraTable)new CassandraTables.IndexLabel());
        }

        @Override
        protected Collection<CassandraTable> tables() {
            ArrayList<CassandraTable> tables = new ArrayList<CassandraTable>(super.tables());
            tables.add(this.counters);
            return tables;
        }

        public void increaseCounter(HugeType type, long increment) {
            this.checkOpened();
            CassandraSessionPool.Session session = ((CassandraStore)this).sessions.session();
            this.counters.increaseCounter(session, type, increment);
        }

        public long getCounter(HugeType type) {
            this.checkOpened();
            CassandraSessionPool.Session session = ((CassandraStore)this).sessions.session();
            return this.counters.getCounter(session, type);
        }

        public boolean isSchemaStore() {
            return true;
        }
    }
}

