/*
 * Decompiled with CFR 0.152.
 */
package org.apache.baremaps.flatgeobuf;

import com.google.common.io.LittleEndianDataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.NoSuchElementException;
import org.apache.baremaps.flatgeobuf.generated.Header;
import org.locationtech.jts.geom.Envelope;

public class PackedRTree {
    private static final int HILBERT_MAX = 65535;
    private static final int NODE_ITEM_LEN = 40;
    private static final String ILLEGAL_NODE_SIZE = "Node size must be at least 2";
    private static final String ILLEGAL_NUMBER_OF_ITEMS = "Number of items must be greater than 0";
    private int numItems;
    private int nodeSize;
    private NodeItem[] nodeItems;
    private long numNodes;
    private List<Pair<Integer, Integer>> levelBounds;

    public PackedRTree(List<Item> items, short nodeSize) {
        this.numItems = items.size();
        this.init(nodeSize);
        int k = (int)(this.numNodes - (long)this.numItems);
        Iterator<Item> it = items.iterator();
        for (int i = 0; i < this.numItems; ++i) {
            this.nodeItems[k++] = it.next().nodeItem();
        }
        this.generateNodes();
    }

    public void init(int nodeSize) {
        if (nodeSize < 2) {
            throw new IllegalArgumentException(ILLEGAL_NODE_SIZE);
        }
        if (this.numItems == 0) {
            throw new IllegalArgumentException(ILLEGAL_NUMBER_OF_ITEMS);
        }
        this.nodeSize = Math.min(nodeSize, 65535);
        this.levelBounds = PackedRTree.generateLevelBounds(this.numItems, this.nodeSize);
        this.numNodes = ((Integer)this.levelBounds.get((int)0).second).intValue();
        this.nodeItems = new NodeItem[Math.toIntExact(this.numNodes)];
    }

    void generateNodes() {
        for (int i = 0; i < this.levelBounds.size() - 1; ++i) {
            long pos = ((Integer)this.levelBounds.get((int)i).first).intValue();
            long end = ((Integer)this.levelBounds.get((int)i).second).intValue();
            long newpos = ((Integer)this.levelBounds.get((int)(i + 1)).first).intValue();
            while (pos < end) {
                NodeItem node = new NodeItem(pos);
                for (long j = 0L; j < (long)this.nodeSize && pos < end; ++j) {
                    node.expand(this.nodeItems[(int)pos++]);
                }
                this.nodeItems[(int)newpos++] = node;
            }
        }
    }

    public static List<Item> hilbertSort(List<Item> items, NodeItem extent) {
        double minX = extent.minX();
        double minY = extent.minY();
        double width = extent.width();
        double height = extent.height();
        items.sort((a, b) -> {
            long hb;
            long ha = PackedRTree.hibert(a.nodeItem, 65535, minX, minY, width, height);
            long delta = ha - (hb = PackedRTree.hibert(b.nodeItem, 65535, minX, minY, width, height));
            if (delta > 0L) {
                return 1;
            }
            if (delta == 0L) {
                return 0;
            }
            return -1;
        });
        return items;
    }

    public static long hibert(NodeItem nodeItem, int hilbertMax, double minX, double minY, double width, double height) {
        long x = 0L;
        long y = 0L;
        if (width != 0.0) {
            x = (long)Math.floor((double)hilbertMax * ((nodeItem.minX() + nodeItem.maxX()) / 2.0 - minX) / width);
        }
        if (height != 0.0) {
            y = (long)Math.floor((double)hilbertMax * ((nodeItem.minY() + nodeItem.maxY()) / 2.0 - minY) / height);
        }
        return PackedRTree.hibert(x, y);
    }

    private static long hibert(long x, long y) {
        long a = x ^ y;
        long b = 0xFFFFL ^ a;
        long c = 0xFFFFL ^ (x | y);
        long d = x & (y ^ 0xFFFFL);
        long A = a | b >> 1;
        long B = a >> 1 ^ a;
        long C = c >> 1 ^ b & d >> 1 ^ c;
        long D = a & c >> 1 ^ d >> 1 ^ d;
        a = A;
        b = B;
        c = C;
        d = D;
        A = a & a >> 2 ^ b & b >> 2;
        B = a & b >> 2 ^ b & (a ^ b) >> 2;
        C ^= a & c >> 2 ^ b & d >> 2;
        D ^= b & c >> 2 ^ (a ^ b) & d >> 2;
        a = A;
        b = B;
        c = C;
        d = D;
        A = a & a >> 4 ^ b & b >> 4;
        B = a & b >> 4 ^ b & (a ^ b) >> 4;
        C ^= a & c >> 4 ^ b & d >> 4;
        D ^= b & c >> 4 ^ (a ^ b) & d >> 4;
        a = A;
        b = B;
        c = C;
        d = D;
        C ^= a & c >> 8 ^ b & d >> 8;
        D ^= b & c >> 8 ^ (a ^ b) & d >> 8;
        a = C ^ C >> 1;
        b = D ^ D >> 1;
        long i0 = x ^ y;
        long i1 = b | 0xFFFFL ^ (i0 | a);
        i0 = (i0 | i0 << 8) & 0xFF00FFL;
        i0 = (i0 | i0 << 4) & 0xF0F0F0FL;
        i0 = (i0 | i0 << 2) & 0x33333333L;
        i0 = (i0 | i0 << 1) & 0x55555555L;
        i1 = (i1 | i1 << 8) & 0xFF00FFL;
        i1 = (i1 | i1 << 4) & 0xF0F0F0FL;
        i1 = (i1 | i1 << 2) & 0x33333333L;
        i1 = (i1 | i1 << 1) & 0x55555555L;
        return i1 << 1 | i0;
    }

    public static NodeItem calcExtent(List<Item> items) {
        return items.stream().map(Item::nodeItem).reduce(new NodeItem(0L), NodeItem::expand);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void write(OutputStream outputStream) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate((int)(40L * this.numNodes));
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        for (NodeItem nodeItem : this.nodeItems) {
            buffer.putDouble(nodeItem.minX());
            buffer.putDouble(nodeItem.minY());
            buffer.putDouble(nodeItem.maxX());
            buffer.putDouble(nodeItem.maxY());
            buffer.putLong(nodeItem.offset());
        }
        buffer.flip();
        try {
            if (buffer.hasRemaining()) {
                byte[] arr = new byte[buffer.remaining()];
                buffer.get(arr);
                outputStream.write(arr);
                outputStream.flush();
            }
        }
        finally {
            buffer.clear();
            buffer = null;
        }
    }

    public static long calcSize(long numItems, int nodeSize) {
        long n;
        if (nodeSize < 2) {
            throw new IllegalArgumentException(ILLEGAL_NODE_SIZE);
        }
        if (numItems == 0L) {
            throw new IllegalArgumentException(ILLEGAL_NUMBER_OF_ITEMS);
        }
        int nodeSizeMin = Math.min(nodeSize, 65535);
        if (numItems > 0x100000000000000L) {
            throw new IndexOutOfBoundsException("Number of items must be less than 2^56");
        }
        long numNodes = n = numItems;
        do {
            n = (n + (long)nodeSizeMin - 1L) / (long)nodeSizeMin;
            numNodes += n;
        } while (n != 1L);
        return numNodes * 40L;
    }

    static List<Pair<Integer, Integer>> generateLevelBounds(int numItems, int nodeSize) {
        int n;
        if (nodeSize < 2) {
            throw new IllegalArgumentException(ILLEGAL_NODE_SIZE);
        }
        if (numItems == 0) {
            throw new IllegalArgumentException(ILLEGAL_NUMBER_OF_ITEMS);
        }
        int numNodes = n = numItems;
        ArrayList<Integer> levelNumNodes = new ArrayList<Integer>();
        levelNumNodes.add(n);
        do {
            n = (n + nodeSize - 1) / nodeSize;
            numNodes += n;
            levelNumNodes.add(n);
        } while (n != 1);
        ArrayList<Integer> levelOffsets = new ArrayList<Integer>();
        n = numNodes;
        Iterator iterator = levelNumNodes.iterator();
        while (iterator.hasNext()) {
            int size = (Integer)iterator.next();
            levelOffsets.add(n -= size);
        }
        LinkedList<Pair<Integer, Integer>> levelBounds = new LinkedList<Pair<Integer, Integer>>();
        for (int i = 0; i < levelNumNodes.size(); ++i) {
            levelBounds.add(new Pair<Integer, Integer>((Integer)levelOffsets.get(i), (Integer)levelOffsets.get(i) + (Integer)levelNumNodes.get(i)));
        }
        return levelBounds;
    }

    public static List<SearchHit> search(ByteBuffer bb, int start, int numItems, int nodeSize, Envelope rect) {
        ArrayList<SearchHit> searchHits = new ArrayList<SearchHit>();
        double minX = rect.getMinX();
        double minY = rect.getMinY();
        double maxX = rect.getMaxX();
        double maxY = rect.getMaxY();
        List<Pair<Integer, Integer>> levelBounds = PackedRTree.generateLevelBounds(numItems, nodeSize);
        int leafNodesOffset = (Integer)levelBounds.get((int)0).first;
        int numNodes = (Integer)levelBounds.get((int)0).second;
        LinkedList<QueueItem> queue = new LinkedList<QueueItem>();
        queue.add(new QueueItem(0L, levelBounds.size() - 1));
        while (!queue.isEmpty()) {
            QueueItem stackItem = (QueueItem)queue.pop();
            int nodeIndex = (int)stackItem.nodeIndex;
            int level = stackItem.level;
            boolean isLeafNode = nodeIndex >= numNodes - numItems;
            int levelEnd = (Integer)levelBounds.get((int)level).second;
            int end = Math.min(nodeIndex + nodeSize, levelEnd);
            int nodeStart = start + nodeIndex * 40;
            for (int pos = nodeIndex; pos < end; ++pos) {
                int offset = nodeStart + (pos - nodeIndex) * 40;
                double nodeMinX = bb.getDouble(offset + 0);
                double nodeMinY = bb.getDouble(offset + 8);
                double nodeMaxX = bb.getDouble(offset + 16);
                double nodeMaxY = bb.getDouble(offset + 24);
                if (maxX < nodeMinX || maxY < nodeMinY || minX > nodeMaxX || minY > nodeMaxY) continue;
                long indexOffset = bb.getLong(offset + 32);
                if (isLeafNode) {
                    searchHits.add(new SearchHit(indexOffset, (long)pos - (long)leafNodesOffset));
                    continue;
                }
                queue.add(new QueueItem(indexOffset, level - 1));
            }
        }
        return searchHits;
    }

    public static SearchResult search(InputStream stream, int start, int numItems, int nodeSize, Envelope rect) throws IOException {
        LittleEndianDataInputStream data = new LittleEndianDataInputStream(stream);
        int dataPos = 0;
        ArrayList<SearchHit> hits = new ArrayList<SearchHit>();
        double minX = rect.getMinX();
        double minY = rect.getMinY();
        double maxX = rect.getMaxX();
        double maxY = rect.getMaxY();
        List<Pair<Integer, Integer>> levelBounds = PackedRTree.generateLevelBounds(numItems, nodeSize);
        int leafNodesOffset = (Integer)levelBounds.get((int)0).first;
        int numNodes = (Integer)levelBounds.get((int)0).second;
        LinkedList<QueueItem> queue = new LinkedList<QueueItem>();
        queue.add(new QueueItem(0L, levelBounds.size() - 1));
        while (!queue.isEmpty()) {
            QueueItem stackItem = (QueueItem)queue.pop();
            int nodeIndex = (int)stackItem.nodeIndex;
            int level = stackItem.level;
            boolean isLeafNode = nodeIndex >= numNodes - numItems;
            int levelBound = (Integer)levelBounds.get((int)level).second;
            int end = Math.min(nodeIndex + nodeSize, levelBound);
            int nodeStart = nodeIndex * 40;
            int skip = nodeStart - dataPos;
            if (skip > 0) {
                PackedRTree.skipNBytes((InputStream)data, skip);
                dataPos += skip;
            }
            for (int pos = nodeIndex; pos < end; ++pos) {
                int offset = nodeStart + (pos - nodeIndex) * 40;
                skip = offset - dataPos;
                if (skip > 0) {
                    PackedRTree.skipNBytes((InputStream)data, skip);
                    dataPos += skip;
                }
                double nodeMinX = data.readDouble();
                dataPos += 8;
                if (maxX < nodeMinX) continue;
                double nodeMinY = data.readDouble();
                dataPos += 8;
                if (maxY < nodeMinY) continue;
                double nodeMaxX = data.readDouble();
                dataPos += 8;
                if (minX > nodeMaxX) continue;
                double nodeMaxY = data.readDouble();
                dataPos += 8;
                if (minY > nodeMaxY) continue;
                long indexOffset = data.readLong();
                dataPos += 8;
                if (isLeafNode) {
                    hits.add(new SearchHit(indexOffset, (long)pos - (long)leafNodesOffset));
                    continue;
                }
                queue.add(new QueueItem(indexOffset, level - 1));
            }
        }
        return new SearchResult(hits, dataPos);
    }

    public static long[] readFeatureOffsets(LittleEndianDataInputStream data, long[] fids, Header header) throws IOException {
        long treeSize = PackedRTree.calcSize((int)header.featuresCount(), header.indexNodeSize());
        List<Pair<Integer, Integer>> levelBounds = PackedRTree.generateLevelBounds((int)header.featuresCount(), header.indexNodeSize());
        long bottomLevelOffset = (long)((Integer)levelBounds.get((int)0).first).intValue() * 40L;
        long pos = 0L;
        long[] featureOffsets = new long[fids.length];
        for (int i = 0; i < fids.length; ++i) {
            if (fids[i] > header.featuresCount() - 1L) {
                throw new NoSuchElementException();
            }
            long nodeItemOffset = bottomLevelOffset + fids[i] * 40L;
            long delta = nodeItemOffset + 32L - pos;
            PackedRTree.skipNBytes((InputStream)data, delta);
            long featureOffset = data.readLong();
            pos += delta + 8L;
            featureOffsets[i] = featureOffset;
        }
        long remainingIndexOffset = treeSize - pos;
        PackedRTree.skipNBytes((InputStream)data, remainingIndexOffset);
        return featureOffsets;
    }

    static void skipNBytes(InputStream stream, long skip) throws IOException {
        for (long remaining = skip; remaining > 0L; remaining -= stream.skip(remaining)) {
        }
    }

    public record NodeItem(double minX, double minY, double maxX, double maxY, long offset) {
        public NodeItem(double minX, double minY, double maxX, double maxY) {
            this(minX, minY, maxX, maxY, 0L);
        }

        public NodeItem(long offset) {
            this(Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY, Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY, offset);
        }

        public double width() {
            return this.maxX - this.minX;
        }

        public double height() {
            return this.maxY - this.minY;
        }

        public static NodeItem sum(NodeItem a, NodeItem b) {
            return a.expand(b);
        }

        public NodeItem expand(NodeItem nodeItem) {
            return new NodeItem(Math.min(nodeItem.minX, this.minX), Math.min(nodeItem.minY, this.minY), Math.max(nodeItem.maxX, this.maxX), Math.max(nodeItem.maxY, this.maxY), this.offset);
        }

        public boolean intersects(NodeItem nodeItem) {
            if (nodeItem.minX > this.maxX) {
                return false;
            }
            if (nodeItem.minY > this.maxY) {
                return false;
            }
            if (nodeItem.maxX < this.minX) {
                return false;
            }
            return !(nodeItem.maxY < this.minY);
        }

        public Envelope toEnvelope() {
            return new Envelope(this.minX, this.maxX, this.minY, this.maxY);
        }
    }

    public record Item(NodeItem nodeItem) {
    }

    public record Pair<T, U>(T first, U second) {
    }

    public record QueueItem(long nodeIndex, int level) {
    }

    public record SearchHit(long offset, long index) {
    }

    public record SearchResult(List<SearchHit> hits, int pos) {
    }
}

