/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.mavibot.btree;

import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.apache.directory.mavibot.btree.AbstractBTree;
import org.apache.directory.mavibot.btree.Addition;
import org.apache.directory.mavibot.btree.BTreeHeader;
import org.apache.directory.mavibot.btree.BTreeTypeEnum;
import org.apache.directory.mavibot.btree.DeleteResult;
import org.apache.directory.mavibot.btree.Deletion;
import org.apache.directory.mavibot.btree.ExistsResult;
import org.apache.directory.mavibot.btree.InMemoryBTreeConfiguration;
import org.apache.directory.mavibot.btree.InMemoryLeaf;
import org.apache.directory.mavibot.btree.InMemoryNode;
import org.apache.directory.mavibot.btree.InMemoryTransactionManager;
import org.apache.directory.mavibot.btree.InsertResult;
import org.apache.directory.mavibot.btree.Modification;
import org.apache.directory.mavibot.btree.ModifyResult;
import org.apache.directory.mavibot.btree.NotPresentResult;
import org.apache.directory.mavibot.btree.Page;
import org.apache.directory.mavibot.btree.ReadTransaction;
import org.apache.directory.mavibot.btree.RemoveResult;
import org.apache.directory.mavibot.btree.SplitResult;
import org.apache.directory.mavibot.btree.Tuple;
import org.apache.directory.mavibot.btree.TupleCursor;
import org.apache.directory.mavibot.btree.exception.InitializationException;
import org.apache.directory.mavibot.btree.exception.KeyNotFoundException;
import org.apache.directory.mavibot.btree.exception.MissingSerializerException;
import org.apache.directory.mavibot.btree.serializer.BufferHandler;
import org.apache.directory.mavibot.btree.serializer.LongSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class InMemoryBTree<K, V>
extends AbstractBTree<K, V>
implements Closeable {
    protected static final Logger LOG = LoggerFactory.getLogger(InMemoryBTree.class);
    public static final String DEFAULT_JOURNAL = "mavibot.log";
    public static final String DATA_SUFFIX = ".db";
    public static final String JOURNAL_SUFFIX = ".log";
    private File file;
    private boolean withJournal;
    private File journal;
    private File envDir;
    private FileChannel journalChannel = null;

    InMemoryBTree() {
        this.setType(BTreeTypeEnum.IN_MEMORY);
    }

    InMemoryBTree(InMemoryBTreeConfiguration<K, V> configuration) {
        String btreeName = configuration.getName();
        if (btreeName == null) {
            throw new IllegalArgumentException("BTree name cannot be null");
        }
        String filePath = configuration.getFilePath();
        if (filePath != null) {
            this.envDir = new File(filePath);
        }
        this.setName(btreeName);
        this.setPageSize(configuration.getPageSize());
        this.setKeySerializer(configuration.getKeySerializer());
        this.setValueSerializer(configuration.getValueSerializer());
        this.setAllowDuplicates(configuration.isAllowDuplicates());
        this.setType(configuration.getType());
        this.readTimeOut = configuration.getReadTimeOut();
        this.writeBufferSize = configuration.getWriteBufferSize();
        if (this.keySerializer.getComparator() == null) {
            throw new IllegalArgumentException("Comparator should not be null");
        }
        BTreeHeader newBtreeHeader = new BTreeHeader();
        newBtreeHeader.setBTreeHeaderOffset(0L);
        newBtreeHeader.setRevision(0L);
        newBtreeHeader.setNbElems(0L);
        newBtreeHeader.setRootPage(new InMemoryLeaf(this));
        newBtreeHeader.setRootPageOffset(0L);
        this.btreeRevisions.put(0L, newBtreeHeader);
        this.currentBtreeHeader = newBtreeHeader;
        try {
            this.init();
        }
        catch (IOException ioe) {
            throw new InitializationException(ioe.getMessage());
        }
    }

    private void init() throws IOException {
        if (this.envDir != null) {
            boolean created;
            if (!this.envDir.exists() && !(created = this.envDir.mkdirs())) {
                throw new IllegalStateException("Could not create the directory " + this.envDir + " for storing data");
            }
            this.file = new File(this.envDir, this.getName() + DATA_SUFFIX);
            this.journal = new File(this.envDir, this.file.getName() + JOURNAL_SUFFIX);
            this.setType(BTreeTypeEnum.BACKED_ON_DISK);
        }
        this.readTransactions = new ConcurrentLinkedQueue();
        this.transactionManager = new InMemoryTransactionManager();
        if (this.getType() == BTreeTypeEnum.BACKED_ON_DISK) {
            if (this.file.length() > 0L) {
                this.load(this.file);
            }
            this.withJournal = true;
            FileOutputStream stream = new FileOutputStream(this.journal);
            this.journalChannel = stream.getChannel();
            if (this.journal.length() > 0L) {
                this.applyJournal();
            }
        } else {
            this.setType(BTreeTypeEnum.IN_MEMORY);
            BTreeHeader btreeHeader = new BTreeHeader();
            btreeHeader.setRootPage(new InMemoryLeaf(this));
            btreeHeader.setBtree(this);
            this.storeRevision(btreeHeader);
        }
    }

    @Override
    protected ReadTransaction<K, V> beginReadTransaction() {
        BTreeHeader btreeHeader = this.getBtreeHeader();
        ReadTransaction readTransaction = new ReadTransaction(btreeHeader, this.readTransactions);
        this.readTransactions.add(readTransaction);
        return readTransaction;
    }

    @Override
    protected ReadTransaction<K, V> beginReadTransaction(long revision) {
        BTreeHeader btreeHeader = this.getBtreeHeader(revision);
        if (btreeHeader != null) {
            ReadTransaction readTransaction = new ReadTransaction(btreeHeader, this.readTransactions);
            this.readTransactions.add(readTransaction);
            return readTransaction;
        }
        return null;
    }

    @Override
    public void close() throws IOException {
        if (this.getType() == BTreeTypeEnum.BACKED_ON_DISK) {
            this.flush();
            this.journalChannel.close();
        }
    }

    @Override
    protected Tuple<K, V> delete(K key, V value, long revision) throws IOException {
        if (revision == -1L) {
            revision = this.currentRevision.get() + 1L;
        }
        BTreeHeader oldBtreeHeader = this.getBtreeHeader();
        BTreeHeader newBtreeHeader = this.createNewBtreeHeader(oldBtreeHeader, revision);
        newBtreeHeader.setBtree(this);
        Tuple tuple = null;
        DeleteResult<K, V> result = this.getRootPage().delete(key, value, revision);
        if (result instanceof NotPresentResult) {
            return null;
        }
        if (result instanceof RemoveResult) {
            RemoveResult removeResult = (RemoveResult)result;
            Page modifiedPage = removeResult.getModifiedPage();
            newBtreeHeader.setRootPage(modifiedPage);
            tuple = removeResult.getRemovedElement();
        }
        if (this.withJournal) {
            this.writeToJournal(new Deletion(key));
        }
        if (tuple != null) {
            newBtreeHeader.decrementNbElems();
        }
        this.storeRevision(newBtreeHeader);
        if (oldBtreeHeader.getNbUsers() == 0) {
            this.btreeRevisions.remove(oldBtreeHeader.getRevision());
        }
        return tuple;
    }

    @Override
    InsertResult<K, V> insert(K key, V value, long revision) throws IOException {
        long oldRevision;
        if (revision == -1L) {
            revision = this.currentRevision.get() + 1L;
        }
        BTreeHeader oldBtreeHeader = this.getBtreeHeader();
        BTreeHeader newBtreeHeader = this.createNewBtreeHeader(oldBtreeHeader, revision);
        newBtreeHeader.setBtree(this);
        Object modifiedValue = null;
        InsertResult result = newBtreeHeader.getRootPage().insert(key, value, revision);
        if (result instanceof ExistsResult) {
            return result;
        }
        if (result instanceof ModifyResult) {
            ModifyResult modifyResult = (ModifyResult)result;
            Page modifiedPage = modifyResult.getModifiedPage();
            newBtreeHeader.setRootPage(modifiedPage);
            modifiedValue = modifyResult.getModifiedValue();
        } else {
            SplitResult splitResult = (SplitResult)result;
            Object pivot = splitResult.getPivot();
            Page leftPage = splitResult.getLeftPage();
            Page rightPage = splitResult.getRightPage();
            newBtreeHeader.setRootPage(new InMemoryNode(this, revision, pivot, leftPage, rightPage));
        }
        if (this.withJournal) {
            this.writeToJournal(new Addition<K, V>(key, value));
        }
        if (modifiedValue == null) {
            newBtreeHeader.incrementNbElems();
        }
        this.storeRevision(newBtreeHeader);
        if (oldBtreeHeader.getNbUsers() == 0 && (oldRevision = oldBtreeHeader.getRevision()) < newBtreeHeader.getRevision()) {
            this.btreeRevisions.remove(oldBtreeHeader.getRevision());
        }
        return result;
    }

    private void writeBuffer(FileChannel channel, ByteBuffer bb, byte[] buffer) throws IOException {
        int size = buffer.length;
        int pos = 0;
        do {
            if (bb.remaining() >= size) {
                bb.put(buffer, pos, size);
                size = 0;
                continue;
            }
            int len = bb.remaining();
            size -= len;
            bb.put(buffer, pos, len);
            pos += len;
            bb.flip();
            channel.write(bb);
            bb.clear();
        } while (size > 0);
    }

    public void flush(File file) throws IOException {
        File parentFile = file.getParentFile();
        File baseDirectory = null;
        baseDirectory = parentFile != null ? new File(file.getParentFile().getAbsolutePath()) : new File(".");
        File tmpFileFD = File.createTempFile("mavibot", null, baseDirectory);
        FileOutputStream stream = new FileOutputStream(tmpFileFD);
        FileChannel ch = stream.getChannel();
        ByteBuffer bb = ByteBuffer.allocateDirect(this.writeBufferSize);
        try {
            TupleCursor cursor = this.browse();
            if (this.keySerializer == null) {
                throw new MissingSerializerException("Cannot flush the btree without a Key serializer");
            }
            if (this.valueSerializer == null) {
                throw new MissingSerializerException("Cannot flush the btree without a Value serializer");
            }
            bb.putLong(this.getBtreeHeader().getNbElems());
            while (cursor.hasNext()) {
                Tuple tuple = cursor.next();
                byte[] keyBuffer = this.keySerializer.serialize(tuple.getKey());
                this.writeBuffer(ch, bb, keyBuffer);
                byte[] valueBuffer = this.valueSerializer.serialize(tuple.getValue());
                this.writeBuffer(ch, bb, valueBuffer);
            }
            if (bb.position() > 0) {
                bb.flip();
                ch.write(bb);
            }
            ch.force(true);
            ch.close();
        }
        catch (KeyNotFoundException knfe) {
            knfe.printStackTrace();
            throw new IOException(knfe.getMessage());
        }
        File backupFile = File.createTempFile("mavibot", null, baseDirectory);
        file.renameTo(backupFile);
        tmpFileFD.renameTo(file);
        backupFile.delete();
    }

    private void applyJournal() throws IOException {
        if (!this.journal.exists()) {
            throw new IOException("The journal does not exist");
        }
        FileChannel channel = new RandomAccessFile(this.journal, "rw").getChannel();
        ByteBuffer buffer = ByteBuffer.allocate(65536);
        BufferHandler bufferHandler = new BufferHandler(channel, buffer);
        try {
            while (true) {
                Object key;
                byte[] type;
                if ((type = bufferHandler.read(1))[0] == 0) {
                    key = this.keySerializer.deserialize(bufferHandler);
                    Object value = this.valueSerializer.deserialize(bufferHandler);
                    this.insert(key, value, this.getBtreeHeader().getRevision());
                    continue;
                }
                key = this.keySerializer.deserialize(bufferHandler);
                this.delete(key, this.getBtreeHeader().getRevision());
            }
        }
        catch (EOFException eofe) {
            eofe.printStackTrace();
            this.journalChannel.truncate(0L);
            return;
        }
    }

    public void load(File file) throws IOException {
        if (!file.exists()) {
            throw new IOException("The file does not exist");
        }
        FileChannel channel = new RandomAccessFile(file, "rw").getChannel();
        ByteBuffer buffer = ByteBuffer.allocate(65536);
        BufferHandler bufferHandler = new BufferHandler(channel, buffer);
        long nbElems = LongSerializer.deserialize(bufferHandler.read(8));
        boolean isJournalActivated = this.withJournal;
        this.withJournal = false;
        for (long i = 0L; i < nbElems; ++i) {
            Object key = this.keySerializer.deserialize(bufferHandler);
            Object value = this.valueSerializer.deserialize(bufferHandler);
            this.insert(key, value, this.getBtreeHeader().getRevision());
        }
        this.withJournal = isJournalActivated;
    }

    @Override
    public Page<K, V> getRootPage(long revision) throws IOException, KeyNotFoundException {
        return this.getBtreeHeader().getRootPage();
    }

    @Override
    public Page<K, V> getRootPage() {
        return this.getBtreeHeader().getRootPage();
    }

    @Override
    void setRootPage(Page<K, V> root) {
        this.getBtreeHeader().setRootPage(root);
    }

    @Override
    public void flush() throws IOException {
        if (this.getType() == BTreeTypeEnum.BACKED_ON_DISK) {
            this.flush(this.file);
            this.journalChannel.truncate(0L);
        }
    }

    public File getFile() {
        return this.file;
    }

    public void setFilePath(String filePath) {
        if (filePath != null) {
            this.envDir = new File(filePath);
        }
    }

    public File getJournal() {
        return this.journal;
    }

    public boolean isInMemory() {
        return this.getType() == BTreeTypeEnum.IN_MEMORY;
    }

    public boolean isPersistent() {
        return this.getType() == BTreeTypeEnum.IN_MEMORY;
    }

    private void writeToJournal(Modification<K, V> modification) throws IOException {
        if (modification instanceof Addition) {
            byte[] keyBuffer = this.keySerializer.serialize(modification.getKey());
            ByteBuffer bb = ByteBuffer.allocateDirect(keyBuffer.length + 1);
            bb.put((byte)0);
            bb.put(keyBuffer);
            bb.flip();
            this.journalChannel.write(bb);
            byte[] valueBuffer = this.valueSerializer.serialize(modification.getValue());
            bb = ByteBuffer.allocateDirect(valueBuffer.length);
            bb.put(valueBuffer);
            bb.flip();
            this.journalChannel.write(bb);
        } else if (modification instanceof Deletion) {
            byte[] keyBuffer = this.keySerializer.serialize(modification.getKey());
            ByteBuffer bb = ByteBuffer.allocateDirect(keyBuffer.length + 1);
            bb.put((byte)1);
            bb.put(keyBuffer);
            bb.flip();
            this.journalChannel.write(bb);
        }
        this.journalChannel.force(true);
    }

    private BTreeHeader<K, V> createNewBtreeHeader(BTreeHeader<K, V> btreeHeader, long revision) {
        BTreeHeader<K, V> newBtreeHeader = new BTreeHeader<K, V>();
        newBtreeHeader.setBTreeHeaderOffset(btreeHeader.getBTreeHeaderOffset());
        newBtreeHeader.setRevision(revision);
        newBtreeHeader.setNbElems(btreeHeader.getNbElems());
        newBtreeHeader.setRootPage(btreeHeader.getRootPage());
        return newBtreeHeader;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        switch (this.getType()) {
            case IN_MEMORY: {
                sb.append("In-memory ");
                break;
            }
            case BACKED_ON_DISK: {
                sb.append("Persistent ");
                break;
            }
            default: {
                sb.append("Wrong type... ");
            }
        }
        sb.append("BTree");
        sb.append("[").append(this.getName()).append("]");
        sb.append("( pageSize:").append(this.getPageSize());
        if (this.getBtreeHeader().getRootPage() != null) {
            sb.append(", nbEntries:").append(this.getBtreeHeader().getNbElems());
        } else {
            sb.append(", nbEntries:").append(0);
        }
        sb.append(", comparator:");
        if (this.keySerializer.getComparator() == null) {
            sb.append("null");
        } else {
            sb.append(this.keySerializer.getComparator().getClass().getSimpleName());
        }
        sb.append(", DuplicatesAllowed: ").append(this.isAllowDuplicates());
        if (this.getType() == BTreeTypeEnum.BACKED_ON_DISK) {
            try {
                sb.append(", file : ");
                if (this.file != null) {
                    sb.append(this.file.getCanonicalPath());
                } else {
                    sb.append("Unknown");
                }
                sb.append(", journal : ");
                if (this.journal != null) {
                    sb.append(this.journal.getCanonicalPath());
                } else {
                    sb.append("Unkown");
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        sb.append(") : \n");
        sb.append(this.getRootPage().dumpPage(""));
        return sb.toString();
    }
}

